(***********************************************************************

                    Mathematica-Compatible Notebook

This notebook can be used on any computer system with Mathematica 3.0,
MathReader 3.0, or any compatible application. The data for the notebook 
starts with the line of stars above.

To get the notebook into a Mathematica-compatible application, do one of 
the following:

* Save the data starting with the line of stars above into a file
  with a name ending in .nb, then open the file inside the application;

* Copy the data starting with the line of stars above to the
  clipboard, then use the Paste menu command inside the application.

Data for notebooks contains only printable 7-bit ASCII and can be
sent directly in email or through ftp in text mode.  Newlines can be
CR, LF or CRLF (Unix, Macintosh or MS-DOS style).

NOTE: If you modify the data for this notebook not in a Mathematica-
compatible application, you must delete the line below containing the 
word CacheID, otherwise Mathematica-compatible applications may try to 
use invalid cache data.

For more information on notebooks and Mathematica-compatible 
applications, contact Wolfram Research:
  web: http://www.wolfram.com
  email: info@wolfram.com
  phone: +1-217-398-0700 (U.S.)

Notebook reader applications are available free of charge from 
Wolfram Research.
***********************************************************************)

(*CacheID: 232*)


(*NotebookFileLineBreakTest
NotebookFileLineBreakTest*)
(*NotebookOptionsPosition[     62517,       1789]*)
(*NotebookOutlinePosition[     86689,       2662]*)
(*  CellTagsIndexPosition[     86645,       2658]*)
(*WindowFrame->Normal*)



Notebook[{

Cell[CellGroupData[{
Cell["Simulating Bugs in Feynman's Quantum Computer", "Title",
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["Colin P. Williams & Scott H. Clearwater", "Subsubtitle"],

Cell["\<\
Needs[\"Statistics`ContinuousDistributions`\"]; 
Needs[\"Statistics`DiscreteDistributions`\"]; 
Needs[\"Statistics`DataManipulation`\"];
Needs[\"Graphics`Graphics`\"];
Off[General::spell];
Off[General::spell1];\
\>", "Input",
  InitializationCell->True],

Cell[CellGroupData[{

Cell["Copyright Notice", "Section",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True],

Cell["\<\
Copyright Colin P. Williams & Scott H. Clearwater (1997).

This Notebook is intended to be used in conjunction with \"Explorations in \
Quantum Computing\" by Colin P. Williams and Scott H. Clearwater, TELOS, \
Springer-Verlag (1997), ISBN:0-387-94768-X. Permission is hereby granted to \
copy and distribute this Notebook freely for any non-commercial activity \
provided you include this copyright notice at the beginning of all such \
copies. Please send suggestions and bug reports to Colin P. Williams at 
        colin@solstice.jpl.nasa.gov      (818) 306 6512 or 
        cpw@cs.stanford.edu               (415) 728 2118
For information on \"Explorations in Quantum Computing\" check out the TELOS \
web site:  http://www.telospub.com/catalog/PHYSICS/Explorations.html. To \
order call 1-800-777-4643.

All other rights reserved.\
\>", "Text",
  Editable->False,
  Evaluatable->False,
  AspectRatioFixed->True]
}, Open  ]],

Cell[CellGroupData[{

Cell["Introduction", "Section"],

Cell["\<\
This Notebook contains code that allows you to experiment with the \
effects of imperfections on the performance of Feynman's quantum computer \
(see \"Explorations in Quantum Computing\" Chapter 5).  An imperfection, such \
as an atom being slightly misaligned, will cause the actual Hamiltonian of \
the quantum computer, Hbuggy, to differ slightly from the intended \
Hamiltonian, H. This in turn will induce an error in the unitary evolution \
operator, U, through the relationship U=Exp[I H t/hBar] (where hBar is \
Planck's constant divided by 2 Pi). The unitary evolution operator is really \
the \"program\" that the quantum computer is following. Thus if the form of \
the actual unitary evolution operator is Ubuggy = Exp[I Hbuggy t/hBar] rather \
than U, this means that the quantum computer will perform a computation other \
than the one intended. However, if the errors in the Hamiltonian are not too \
great then it is possible that the quantum computer will still perform the \
correct computation accidentally. The question is how bad can the errors be \
before the output from the quantum computer is essentially useless? The code \
in this Notebook helps to answer this question.\
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell[">>> README >>>", "Section"],

Cell[TextData[{
  "If you are using ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " version 2.2 you must open and initialize all input cells in the Notebook \
Feynman.ma before the code in this Notebook (QBugs.ma) will work. If you are \
using ",
  StyleBox["Mathematica",
    FontSlant->"Italic"],
  " v3.0 use Feynman.nb and QBugs.nb instead. Both Feynman.ma(.nb) and \
QBugs.ma(.nb) must be open and initialized before the code for simulating \
bugs in Feynman's quantum computer will work properly."
}], "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Circuit for NOT Built From Two SqrtNOT Gates", "Section"],

Cell["\<\
We are going to use our good old sqrtNOTcircuit that was introduced \
in Chapter 4. This circuit computes the NOT function as two SqrtNOT gates \
connected back to back. The code that defines the circuit is contained in the \
Feynman.ma(.nb) Notebook. The circuit is represented as a pair of square root \
of NOT gates.\
\>", "Text"],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection"],

Cell["\<\
Here is the definition of the sqrtNOTcircuit. To see the circuit \
representation click on \"sqrtNOTcircuit\" in the cell below and hit the \
RETURN key whilst holding down the SHIFT key simultaneously:\
\>", "Text"],

Cell[BoxData[
    \(sqrtNOTcircuit\)], "Input"],

Cell["\<\
If you do not see a list of two matrices when you evaluated \
sqrtNOTcircuit it means that you have not initialized the Feynman.ma(.nb) \
Notebook. In version 3.0 you initialize a Notebook by using the Kernel menu \
and selecting the Kernel/Evaluation/Evaluate Initialization sub-menu. In \
version 2.2 you initialize a Notebook using the Action menu by selecting the \
Action/Evaluate Initialization sub-menu.\
\>", "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Correct Behavior of the Quantum Computer", "Section"],

Cell["\<\
If the computation to be performed is NOT computed as two SqrtNOT \
gates connected back to back then we have SqrtNOT(SqrtNOT(x)) = x. If the \
input bit is x, the input state (which includes the auxiliary cursor bits) is \
the state ket[1,0,0,x]: the first three qubits represent the cursor and the \
last qubit, x, represents the input bit. After the quantum computer has \
evolved the memory register ought to evolve into the state ket[0,0,1,NOT(x)]. \
Note that the cursor advances from left to right in such a way that there is \
only ever one cursor bit set to 1. Thus if the input state is ket[1,0,0,0] \
the output should be ket[0,0,1,1]. Conversely, if the input state is \
ket[1,0,0,1] the output state ought to be ket[0,0,1,0]. The sqrtNOTcircuit \
can handle superpositions of inputs but we will restrict attention to just \
classical inputs throughout this Notebook. The reason we do this is because \
you cannot assess the correctness of a quantum computation on a superposed \
input state in a single run. To test that would need to do several runs and \
compare the distribution of \"answers\" (note plural) that  you see with the \
distribution of answers that you expect. So for our purposes we will use only \
classical inputs and expect only classical outputs.

We regard the computation as correct if both the answer bit is \"correct\" \
and the cursor is \"uncorrupted\". By a \"correct\" answer bit we mean, if \
the input to the computer was the binary digit x then, when the cursor bit \
was found to be in its terminal (i.e. third) position, the output memory \
register bit was NOT(x). By an \"uncorrupted\" cursor we mean at those prior \
times when the cursor was measured and found not to be at its terminal \
position, there was at most one 1 amongst the three cursor bits. Thus by \
restricting attention to classical binary inputs (0 or 1 and not \
superpositions of 0 and 1) we can determine whether the computation is \
correct simply by reading the cursor bits (at every time step) and the answer \
bit (at the last time step when the cursor is at its terminal position).\
\>",
   "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Modeling Errors in the Hamiltonian", "Section"],

Cell["\<\
The Hamiltonian of a quantum computer is described mathematically \
by a hermitian matrix. This is a matrix whose (i,j)-th element is the complex \
conjugate of its (j,i)-th element. Even the \"buggy\" Hamiltonian \
representing a faulty computation must be described by some hermitian matrix. \
So whetever we do to create a buggy Hamiltonian from the correct Hamiltonian, \
we had better be sure that the operation preserves hermiticity. 

To model errors in the Hamiltonian, we start by computing the correct \
Hamiltonian for the desired circuit (i.e. computation) using the function \
Hamiltonian defined in the Feynman.ma(.nb) Notebook, and then adding random \
normally distributed noise to the upper triangular elements of the \
Hamiltonian. We then set the lower triangular elements to be the complex \
conjugates of the \"buggy\" upper triangular elements and voila! we have a \
buggy Hamiltonian. By assumption we set the mean error to be 0 and the \
standard deviation in the error to be stdDevError. Thus stdDevError is the \
parameter that determines how far we nudge the elements of the Hamiltonian \
away from their indented values. A reasonable value to use for stdDevError \
(if you want to mimic the effects of small to medium errors) is to have 0 <= \
stdDevError <= 1. 

This procedure gives us a hermitian matrix (buggy Hamiltonian) that is \
\"close\" to the correct hermitian matrix (correct Hamiltonian). However, \
there are certainly other more physically motivated ways to generate a buggy \
Hamiltonian. Nevertheless, our simple mathematical is adequate to illustrate \
how errors grow due to imperfections.\
\>", "Text"],

Cell["\<\
BuggyHamiltonian[m_, k_, circuit_, stdDevError_] :=
  Module[{ham}, 
  \tham = Hamiltonian[m, k, circuit]; 
    For[i = 1, i <= 2^(m + k + 1), i++, 
     For[j = i, j <= 2^(m + k + 1), j++, 
       ham[[i,j]] = 
         ham[[i,j]] + Abs[Random[NormalDistribution[0., stdDevError]]] + 
          I*Random[NormalDistribution[0., stdDevError]]; 
        ham[[j,i]] = Conjugate[ham[[i,j]]]; ]; ]; 
    ham
  ]\
\>", "Input",
  InitializationCell->True],

Cell["\<\
A buggy Hamiltonian automatically induces a buggy unitary evolution \
operator. So we will overwrite the old definition of EvolutionOP in our \
Feynman.nb simulator. The new definition of EvolutionOP uses the buggy \
Hamiltonian.\
\>", "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Simulating Bugs in Feyman's Quantum Computer", "Section"],

Cell["\<\
We are now ready to try to simulate bugs in Feynman's quantum \
computer. The top level function is SimulateBugsInFeynmanQC. This function \
takes 6 arguments:

(1) The initial state fed into the quantum computer.
(2) The correct output state.
(3) The quantum circuit (i.e. the computation to be performed)
(4) The number of runs of the simulator using this input and this circuit.
(5) The standard deviation of the random \"noise\" terms added to the \
Hamiltonian matrix elements
    (typically between 0 and 1)
(6) The name of a file in which to append the output data.\
\>", "Text"],

Cell["\<\
The output is a 10-argument list:
(A) The standard deviation of the errors that was used (this was input 5 \
above)
(B) Number of steps to termination for those runs that yielded a correct \
computation (right answer/good cursor)
(C) The mean or (B)
(D) The standard deviation of (B)
(E) Number of steps to termination for those runs that yielded a faulty \
computation (wrong answer/good cursor)
(F) The mean of (E)
(G) The standard deviation of (E)
(H) Number of steps to termination for those runs that yielded an unreliable \
computation (bad cursor)
(I) The mean of (H)
(J) The standard deviation of (H)\
\>", "Text"],

Cell["\<\
If you forget what the inputs are ask for the usage of \
SimulateBugsInFeynmanQC.\
\>", "Text"],

Cell[CellGroupData[{

Cell["?SimulateBugsInFeynmanQC", "Input"],

Cell["\<\
SimulateBugsInFeynmanQC[initState,correctAns,circuit,n,stdDevError,\
file] performs n
   independent runs of a Feynman quantum computer for the given circuit \
starting from
   the given initial state. The Hamiltonian for this circuit is \"corrupted\" \
by
   introducing random normally distributed errors to its matrix elements. The \
errors
   have a mean of zero and a standard deviation of stdDevError. The errors \
are
   introduced in such a way as to keep the \"buggy\" Hamiltonian hermitian. \
The output is
   a 10 element list that shows the stdDevError, the number of steps to \
termination of
   runs that ended with the correct answer (and legal cursor), their mean and \
standard
   deviation, the number of steps to termination for runs that ended with the \
wrong
   answer (but legal cursor), their mean and standard deviation and the \
number of steps
   to termination for runs that ended with an illegal cursor (i.e. a cursor \
with more
   than one 1) whether the answer was correct or incorrect and their mean and \
standard
   deviation. The output is printed to the screen and recorded in the given \
file.\
\>", "Print"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try Me!", "Subsubsection"],

Cell["\<\
Let's begin by seeing what happens when we ask for 25 runs of a \
Feynman quantum computer used to compute the NOT of 1 when the standard \
deviation in the noise added to the Hamiltonian is 0.05. Remember you must \
open and evaluate the initialization cells in Feynman.ma(.nb) in order for \
SimulateBugsInFeynmanQC code to work.

The input state contains the initial specification of the cursor, {1,0,0}, \
together with the input to the circuit, the binary value 1. We enter the 4 \
bits {1,0,0,1} as the 4-particle quantum state ket[1,0,0,1].  The computation \
is supposed to be \"complete and correct\" whenever the cursor is observed to \
be in its terminal position (the 1 appears on the (k+1)-th qubit. Thus if we \
measure the state of just the cursor qubits (the leftmost k+1=3 qubits in our \
current case) and find them in the state ket[0,0,1,?x] where ?x is \
unmeasured, we know that at that moment a measurement of ?x ought to reveal \
the correct answer to the computation, i.e. ?x ought to be 0. Therefore, if \
the input state is ket[1,0,0,1] and the computer is performing the NOT \
computation correctly, the entire memory register should evolve into the \
state ket[0,0,1,0] . Try the following example and see what you get.\
\>", 
  "Text"],

Cell[CellGroupData[{

Cell["\<\
SimulateBugsInFeynmanQC[ket[1,0,0,1],
                        ket[0,0,1,0],
                        sqrtNOTcircuit,
                        25,
                        0.05,
                        \"test.dat\"]\
\>", "Input"],

Cell["\<\
Simulating errors at the level stdDevError = 0.05
Output data is appended to the file test.dat
Number of gates in circuit, k = 2
Number of qubits in cursor, k+1 = 3
Number of qubits for input to/output from circuit, m = 1
Total number of qubits in memory register m+k+1 = 4
Number of runs = 25
Completed 12 runs out of 25
\tFraction of runs that are corrupted i.e. have bad cursor = 0.0833333
Completed 24 runs out of 25
\tFraction of runs that are corrupted i.e. have bad cursor = 0.166667
CPU time for simulation = 456.00000 seconds



Fraction of runs that were successful
\t(i.e. gave correct answer and had legal cursor) = 0.72
Average time to complete a successful run = 3.11111 +/- 2.69834

Fraction of runs that were unsuccessful
\t(i.e. gave wrong answer but had legal cursor) = 0.12
Average time to complete an unsuccessful run = 5.66667 +/- 3.51188

Fraction of runs that were inconclusive
\tbecause they had a bad cursor = 0.16
Average time to detect bad cursor = 1.5 +/- 1.\
\>", "Print"],

Cell[OutputFormData[
"\<\
{0.05, {1, 3, 1, 4, 3, 1, 6, 12, 2, 1, 2, 2, 6, 3, 2, 1, 3, 3}, \
3.111111111111111, 
  2.698341296358558, {6, 2, 9}, 5.666666666666666, 3.511884584284246, {3, 1, \
1, 1}, 
  1.5, 1.}\
\>", 
"\<\
{0.05, {1, 3, 1, 4, 3, 1, 6, 12, 2, 1, 2, 2, 6, 3, 2, 1, 3, 3}, 3.11111, \
2.69834, 
 
  {6, 2, 9}, 5.66667, 3.51188, {3, 1, 1, 1}, 1.5, 1.}\
\>"], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection"],

Cell["\<\
If you increase the magnitude of the errors in the Hamiltonian the \
computation becomes more unreliable. The following example illustrates what \
happens when the standard deviation of the error in the Hamiltonian is 0.5. \
Note that if we use the same filename as above, \"test.dat\", the new answers \
(with stdDevError = 0.5) will be appended to the end of the file.\
\>", "Text"],

Cell[CellGroupData[{

Cell["\<\
SimulateBugsInFeynmanQC[ket[1,0,0,1],
                        ket[0,0,1,0],
                        sqrtNOTcircuit,
                        25,
                        0.5,
                        \"test.dat\"]\
\>", "Input"],

Cell["\<\
Simulating errors at the level stdDevError = 0.5
Output data is appended to the file test.dat
Number of gates in circuit, k = 2
Number of qubits in cursor, k+1 = 3
Number of qubits for input to/output from circuit, m = 1
Total number of qubits in memory register m+k+1 = 4
Number of runs = 25
Completed 12 runs out of 25
\tFraction of runs that are corrupted i.e. have bad cursor = 0.666667
Completed 24 runs out of 25
\tFraction of runs that are corrupted i.e. have bad cursor = 0.708333
CPU time for simulation = 179.00000 seconds



Fraction of runs that were successful
\t(i.e. gave correct answer and had legal cursor) = 0.16
Average time to complete a successful run = 1. +/- 0

Fraction of runs that were unsuccessful
\t(i.e. gave wrong answer but had legal cursor) = 0.12
Average time to complete an unsuccessful run = 1.33333 +/- 0.57735

Fraction of runs that were inconclusive
\tbecause they had a bad cursor = 0.72
Average time to detect bad cursor = 1.11111 +/- 0.323381\
\>", "Print"],

Cell[OutputFormData[
"\<\
{0.5, {1, 1, 1, 1}, 1., 0, {2, 1, 1}, 1.333333333333333, \
0.5773502691896258, 
  {1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 1, 1}, 1.111111111111111, 
  0.3233808333817772}\
\>", 
"\<\
{0.5, {1, 1, 1, 1}, 1., 0, {2, 1, 1}, 1.33333, 0.57735, 
 
  {1, 1, 1, 1, 1, 1, 1, 1, 1, 2, 1, 1, 2, 1, 1, 1, 1, 1}, 1.11111, \
0.323381}\
\>"], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Code for SimulateBugsInFeynmanQC in here ...", "Subsubsection"],

Cell["\<\
Options[SimulateBugsInFeynmanQC] = {Report -> True};

SimulateBugsInFeynmanQC[(w_.)*ket[bits__] + (kets_.),correctAns_,circuit_,
                        nRuns_, stdDevError_, filename_, opts___] := 
  Module[{reportQ, m, k, x, starttime, ham, evoln, timesRightAnsGoodCur,
          timesWrongAnsGoodCur,timesNoAnsBadCur, runs,  stream,
          meanR, meanW, meanB, stdDevR, stdDevW, stdDevB}, 
    reportQ = Report /. {opts} /. Options[SimulateBugsInFeynmanQC];
    Off[EvolveQC::badCursor]; (* turn off error message *)
  \tClear[$hamiltonian];
  \tPrint[\"Simulating errors at the level stdDevError = \", stdDevError];
  \t
  \tIf[reportQ, Print[\"Output data is appended to the file \", filename]];
  \tk = Length[circuit];
  \tIf[reportQ, 
  \t   (Print[\"Number of gates in circuit, k = \", k];
  \t    Print[\"Number of qubits in cursor, k+1 = \", k+1]
  \t   )];
  \tm = Length[{bits}] - (k + 1); 
  \tIf[reportQ,
  \t   (Print[\"Number of qubits for input to/output from circuit, m = \", \
m];
  \t    Print[\"Total number of qubits in memory register m+k+1 = \", m+k+1]
  \t   )];
    x = {}; 
    If[reportQ, Print[\"Number of runs = \", nRuns]]; 
    starttime = AbsoluteTime[]; 
    For[runs = 1, runs <= nRuns, runs++, 
     $hamiltonian = BuggyHamiltonian[m, k, circuit, stdDevError];
     (Clear[EvolutionOP];
      EvolutionOP[_, _, _, t_]:=
      \tMatrixPower[N[MatrixExp[-(I $hamiltonian)]], t]
     );
     evoln = EvolveQC[w*ket[bits] + kets, circuit]; 
     x = Append[x, Last[evoln]];
     If[Mod[runs, If[nRuns<10, 2, Round[nRuns/2]]] == 0,
        Print[\"Completed \", runs, \" runs out of \", nRuns];
        Print[\"\\tFraction of runs that are corrupted i.e. have bad cursor = \
\",
              N[Length[Select[x, (#1[[3]] == $BadCursor)&]]/Length[x]]]; 
         ]; ]; 
    Print[\"CPU time for simulation = \",
          AbsoluteTime[] - starttime, \" seconds\", \"\\n\"]; 
          
    {timesRightAnsGoodCur, meanR, stdDevR, 
     timesWrongAnsGoodCur, meanW, stdDevW,
     timesNoAnsBadCur,     meanB, stdDevB}=
     \tReadAnswerAndSummarizeRuns[x, correctAns, Report -> reportQ]; 
    
    stream = OpenAppend[filename]; 
    Write[stream, {stdDevError, 
                   timesRightAnsGoodCur, meanR, stdDevR,
                   timesWrongAnsGoodCur, meanW, stdDevW,
                   timesNoAnsBadCur, meanB, stdDevB}]; 
    Close[stream];

\t{stdDevError, 
     timesRightAnsGoodCur, meanR, stdDevR,
     timesWrongAnsGoodCur, meanW, stdDevW,
     timesNoAnsBadCur, meanB, stdDevB}
  ]

SimulateBugsInFeynmanQC::usage = 
\"SimulateBugsInFeynmanQC[initState,correctAns,circuit,n,stdDevError,file] \\
performs n independent runs of a Feynman quantum computer for the \\
given circuit starting from the given initial state. The Hamiltonian \\
for this circuit is \\\"corrupted\\\" by introducing random normally \\
distributed errors to its matrix elements. The errors have a mean \\
of zero and a standard deviation of stdDevError. The errors are \\
introduced in such a way as to keep the \\\"buggy\\\" Hamiltonian \\
hermitian. The output is a 10 element list that shows the stdDevError, \\
the number of steps to termination of runs that ended with the correct \\
answer (and legal cursor), their mean and standard deviation, the \\
number of steps to termination for runs that ended with the \\
wrong answer (but legal cursor), their mean and standard deviation \\
and the number of steps to termination for runs that ended with an \\
illegal cursor (i.e. a cursor with more than one 1) whether the answer was \
correct or incorrect and their mean and standard deviation. The output \\
is printed to the screen and recorded in the given file.\";\
\>", "Input",
  InitializationCell->True],

Cell["\<\
ReadAnswerAndSummarizeRuns analyzes the final descriptions of n \
runs and creates summary statistics which can be appended to a file of your \
choice. We regard the answer as correct if (a) the cursor bits contain at \
most one bit set to \"1\" and (b) the answer in the memory register is indeed \
the correct answer to the computation. Note that in order to be able to \
assess the correctness of the answer, we restrict ourselves to running \
simulations of computations that are function evaluations over classical \
inputs.  Our Feynman quantum computer will work on superpositions of inputs \
too but the analysis of the errors becomes more complicated. Note however, in \
the case of classical inputs were we expect only one legitimate answer we can \
compare the predicted output state to the actual output state. The \
computation is deemed correct and reliable if they are the same up to an \
arbitrary phase factor i.e. the following test returns True.\
\>", "Text"],

Cell[CellGroupData[{

Cell["MatchQ[actualOutputState, _. predictedOutputState]", "Input"],

Cell[OutputFormData["\<\
True\
\>", "\<\
True\
\>"], "Output"]
}, Open  ]],

Cell["\<\
ReadAnswerAndSummarizeRuns reads a single output from each \
independent run of the quantum computer and records statistics on the the \
number of times the computation ended with the correct answer, the incorrect \
answer or an unreliable answer. Each run of the quantum computer is \
supposedly doing the \"same\" computation with different but equally buggy \
Hamiltonians. Note that ReadAnswerAnsSummarizeRuns is non-deterministic: two \
identical calls to ReadAnswerAndSummarizeRuns can result in different outputs \
due to the essential nondeterminsin inherent in the quantum mechanics of \
measurement on states that are superpositions of possible answers.\
\>", 
  "Text"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Code for ReadAnswerAndSummarizeRuns in here ...", "Subsubsection"],

Cell["\<\
Options[ReadAnswerAndSummarizeRuns] = {Report -> False};

ReadAnswerAndSummarizeRuns[terminalDescriptions_, correctAns_, opts___]:=
\tModule[{reportQ,pairs,tRAnsGCur,tWAnsGCur,tBadCur, mr,sr,mw,sw,mb,sb},
\t  reportQ   = Report /. {opts} /. Options[ReadAnswerAndSummarizeRuns];
\t  pairs     = Map[{#, ReadMemoryRegister[ExtractFinalState[#]][[2]]}&,
\t\t          terminalDescriptions];
\t  tRAnsGCur = Map[ExtractTime[#[[1]]]&, 
\t                  RightAnsAndGoodCursor[pairs, correctAns]];
\t  tWAnsGCur = Map[ExtractTime[#[[1]]]&, 
\t                  WrongAnsAndGoodCursor[pairs, correctAns]];
\t  tBadCur   = Map[ExtractTime[#[[1]]]&, 
\t                  UnreliableAnsAndBadCursor[pairs]];
\t  mr        =  N[myMean[tRAnsGCur]]; sr = N[myStdDev[tRAnsGCur]];
   \t  mw        =  N[myMean[tWAnsGCur]]; sw = N[myStdDev[tWAnsGCur]];
   \t  mb        =  N[myMean[tBadCur]];   sb = N[myStdDev[tBadCur]];
   If[reportQ===True,
\t(ReportFractionRightAnsGoodCursor[{tRAnsGCur,tWAnsGCur,tBadCur},mr,sr];
     ReportFractionWrongAnsGoodCursor[{tRAnsGCur,tWAnsGCur,tBadCur},mw,sw];
     ReportFractionBadCursor[{tRAnsGCur,tWAnsGCur,tBadCur},mb,sb]
    )
   ];
    {tRAnsGCur, mr, sr, 
     tWAnsGCur, mw, sw, 
     tBadCur,   mb, sb}               
   ]
\t

RightAnsAndGoodCursor[pairs_, correctAns_]:=
\tSelect[Select[pairs, MatchQ[#[[2]], _. correctAns]&],
\t       GoodCursorQ[ExtractCursor[#[[1]]]]&]

WrongAnsAndGoodCursor[pairs_, correctAns_]:=
\tSelect[Select[pairs, Not[MatchQ[#[[2]], _. correctAns]]&],
\t       GoodCursorQ[ExtractCursor[#[[1]]]]&]

UnreliableAnsAndBadCursor[pairs_]:=
\tSelect[pairs, (ExtractCursor[#[[1]]] === $BadCursor)&]       

\t       
GoodCursorQ[$BadCursor]:=False
GoodCursorQ[cursorBits_List]:=
\tIf[Count[cursorBits,1]==1, True, False]
\t
ExtractTime[{time_,_,_,_}]:= time
ExtractCursor[{_,_,cursor_,_}]:= cursor
ExtractFinalState[{_,_,_,finalState_}]:= finalState\
\>", "Input",
  InitializationCell->True],

Cell["\<\
ReportFractionRightAnsGoodCursor[{r_, w_, b_}, mean_, stdDev_]:=
\t(Print[\"\\n\"];
\t Print[\"Fraction of runs that were successful\"];
\t Print[\"\\t(i.e. gave correct answer and had legal cursor) = \",
\t       N[Length[r]/Length[Flatten[{r,w,b}]]]];
\t Print[\"Average time to complete a successful run = \", 
\t       mean, \" +/- \", stdDev, \"\\n\"]
\t)
\t
ReportFractionWrongAnsGoodCursor[{r_, w_, b_}, mean_, stdDev_]:=
\t(Print[\"Fraction of runs that were unsuccessful\"];
\t Print[\"\\t(i.e. gave wrong answer but had legal cursor) = \",
\t       N[Length[w]/Length[Flatten[{r,w,b}]]]];
\t Print[\"Average time to complete an unsuccessful run = \", 
\t       mean, \" +/- \", stdDev, \"\\n\"]
\t)

ReportFractionBadCursor[{r_, w_, b_}, mean_, stdDev_]:=
\t(Print[\"Fraction of runs that were inconclusive\"];
\t Print[\"\\tbecause they had a bad cursor = \",
\t       N[Length[b]/Length[Flatten[{r,w,b}]]]];
\t Print[\"Average time to detect bad cursor = \", 
\t       mean, \" +/- \", stdDev]
\t)\
\>", "Input",
  InitializationCell->True],

Cell["\<\
myMean[{}]:=0 (* no data *)
myMean[x_]:=Mean[x]

myStdDev[{}]:=0  (* no data *)
myStdDev[{_}]:=0 (* one data point *)
myStdDev[x_]:=StandardDeviation[x]\
\>", "Input",
  InitializationCell->True]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Creating and Visualizing Buggy Quantum Computations", "Section"],

Cell["\<\
We now have all the machinery necessary to simulate the effects of \
errors in the Hamiltonian of a quantum computer on the correctness and \
reliability of the answers obtained from the computer. The function \
CollectBugStatistics performs several runs of a quantum computer engaged in a \
specific computation as a function of the level of the error in the \
Hamiltonian (stdDevError). You can specify stdDevError to be in the range \
stdDevErrorMin <= stdDevError <= stdDevErrorMax, and to sweep through that \
range in steps of size stdDevErrorIncr.\
\>", "Text"],

Cell[CellGroupData[{

Cell["?CollectBugStatistics", "Input"],

Cell["\<\
CollectBugStatistics[inputState, correctOutputState, circuit, n, \
{stdDevErrorMin,
   stdDevErrorMax, stdDevErrorIncr},file] performs n runs of the computation \
implied by
   the given circuit for each value of the error in the range stdDevErrorMin \
to
   stdDevErrorMax in steps of size stdDevErrorIncr. The output is the set of \
summary
   statistics for each data point as returned by a call to \
SimulateBugsInFeynmanQC for
   a specific value of stdDevError.\
\>", "Print"]
}, Open  ]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection"],

Cell["\<\
Here we explore runs of the NOT computation implemented as two \
square root of NOT gates connected back to back. The input state is \
ket[1,0,0,1] (i.e. a starting cursor 100 and an input bit 1). The correct \
output state is ket[0,0,1,0] (i.e. an ending cursor 001 and an output bit \
NOT(1) = 0). We sweep out values of the error in the Hamiltonian ranging from \
0 to 0.2 in intervals 0.03. There are 15 samples per data point. This number \
is too low for serious work but illustrates how to call CollectBugStatistics. \
all outputs are written to the file logfile.dat.\
\>", "Text"],

Cell[CellGroupData[{

Cell["\<\
bugStatistics =
CollectBugStatistics[ket[1,0,0,1], ket[0,0,1,0], sqrtNOTcircuit, 15,
                     {0, 0.2, 0.03}, \"logfile.dat\"]\
\>", "Input"],

Cell["\<\
Simulating errors at the level stdDevError = 0
Completed 8 runs out of 15
\tFraction of runs that are corrupted i.e. have bad cursor = 0
CPU time for simulation = 207.00000 seconds

Simulating errors at the level stdDevError = 0.03
Completed 8 runs out of 15
\tFraction of runs that are corrupted i.e. have bad cursor = 0
CPU time for simulation = 181.00000 seconds

Simulating errors at the level stdDevError = 0.06
Completed 8 runs out of 15
\tFraction of runs that are corrupted i.e. have bad cursor = 0.125
CPU time for simulation = 202.00000 seconds

Simulating errors at the level stdDevError = 0.09
Completed 8 runs out of 15
\tFraction of runs that are corrupted i.e. have bad cursor = 0.25
CPU time for simulation = 200.00000 seconds

Simulating errors at the level stdDevError = 0.12
Completed 8 runs out of 15
\tFraction of runs that are corrupted i.e. have bad cursor = 0.625
CPU time for simulation = 165.00000 seconds

Simulating errors at the level stdDevError = 0.15
Completed 8 runs out of 15
\tFraction of runs that are corrupted i.e. have bad cursor = 0.125
CPU time for simulation = 178.00000 seconds

Simulating errors at the level stdDevError = 0.18
Completed 8 runs out of 15
\tFraction of runs that are corrupted i.e. have bad cursor = 0.5
CPU time for simulation = 128.00000 seconds
\
\>", "Print"],

Cell[OutputFormData[
"\<\
{{0, {11, 7, 1, 2, 1, 2, 1, 4, 2, 9, 5, 2, 12, 3, 2}, 4.266666666666666, \

   3.731462116089284, {}, 0, 0, {}, 0, 0}, 
  {0.03, {2, 3, 2, 4, 2, 5, 2, 1, 2, 1, 1, 5, 1}, 2.384615384615384, 
   1.445594545418455, {}, 0, 0, {1, 1}, 1., 0}, 
  {0.06, {1, 5, 1, 1, 1, 6, 3, 1, 1, 2, 6}, 2.545454545454545, \
2.114882330704777, 
   {3}, 3., 0, {3, 2, 4}, 3., 1.}, 
  {0.09, {4, 2, 2, 1, 2, 3, 7, 2, 1, 5, 5}, 3.09090909090909, \
1.921173883569389, {2}, 
   2., 0, {1, 1, 1}, 1., 0}, {0.1199999999999999, {1, 2, 2, 1, 1, 1, 5, 7}, \
2.5, 
   2.267786838055362, {3}, 3., 0, {2, 2, 1, 1, 1, 1}, 1.333333333333333, 
   0.5163977794943221}, {0.15, {1, 1, 1, 4, 1, 2, 1}, 1.571428571428571, 
   1.133893419027681, {10, 4, 1}, 5., 4.582575694955839, {1, 1, 3, 1, 2}, \
1.6, 
   0.8944271909999159}, {0.1799999999999999, {1, 1, 1, 1, 1, 2, 2}, \
1.285714285714285, 
   0.4879500364742666, {1}, 1., 0, {1, 1, 1, 3, 2, 1, 3}, 1.714285714285714, 
   0.9511897312113418}}\
\>", 
"\<\
{{0, {11, 7, 1, 2, 1, 2, 1, 4, 2, 9, 5, 2, 12, 3, 2}, 4.26667, 3.73146, \
{}, 0, 0, {}, 
 
   0, 0}, {0.03, {2, 3, 2, 4, 2, 5, 2, 1, 2, 1, 1, 5, 1}, 2.38462, 1.44559, \
{}, 0, 0, 
 
   {1, 1}, 1., 0}, {0.06, {1, 5, 1, 1, 1, 6, 3, 1, 1, 2, 6}, 2.54545, \
2.11488, {3}, 
 
   3., 0, {3, 2, 4}, 3., 1.}, {0.09, {4, 2, 2, 1, 2, 3, 7, 2, 1, 5, 5}, \
3.09091, 
 
   1.92117, {2}, 2., 0, {1, 1, 1}, 1., 0}, 
 
  {0.12, {1, 2, 2, 1, 1, 1, 5, 7}, 2.5, 2.26779, {3}, 3., 0, {2, 2, 1, 1, 1, \
1}, 
 
   1.33333, 0.516398}, {0.15, {1, 1, 1, 4, 1, 2, 1}, 1.57143, 1.13389, {10, \
4, 1}, 5., 
 
   4.58258, {1, 1, 3, 1, 2}, 1.6, 0.894427}, 
 
  {0.18, {1, 1, 1, 1, 1, 2, 2}, 1.28571, 0.48795, {1}, 1., 0, {1, 1, 1, 3, 2, \
1, 3}, 
 
   1.71429, 0.95119}}\
\>"], "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["Code for CollectBugStatistics in here ...", "Subsubsection"],

Cell["\<\
CollectBugStatistics[inState_,correctOutState_,circuit_,nRuns_,
                     {stdDevErrorMin_, stdDevErrorMax_, stdDevErrorIncr_},
                     file_]:=
\tTable[SimulateBugsInFeynmanQC[inState, correctOutState,
                                  circuit, nRuns, stdDevError, file,
                                  Report->False],
          {stdDevError, stdDevErrorMin, stdDevErrorMax, stdDevErrorIncr}]
          
CollectBugStatistics::usage =
\"CollectBugStatistics[inputState, correctOutputState, circuit, n, \
{stdDevErrorMin, stdDevErrorMax, stdDevErrorIncr},file] performs n \\
runs of the computation implied by the given circuit for each value \\
of the error in the range stdDevErrorMin to stdDevErrorMax in steps \\
of size stdDevErrorIncr. The output is the set of summary statistics \\
for each data point as returned by a call to SimulateBugsInFeynmanQC \\
for a specific value of stdDevError.\";\
\>", "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{

Cell["Try me!", "Subsubsection"],

Cell["\<\
To plot the fraction of runs that ended, at time t=2, with an \
uncorrupted cursor and the correct answer to the computation, use \
PlotFractionCorrectAtTimeTVersusStdDevError[bugStatistics, 2]\
\>", "Text"],

Cell[CellGroupData[{

Cell["\<\
PlotFractionCorrectAtTimeTVersusStdDevError[bugStatistics,2];\
\>", 
  "Input"],

Cell[GraphicsData["PostScript", "\<\
%!
%%Creator: Mathematica
%%AspectRatio: .61803 
MathPictureStart
/Mabs {
Mgmatrix idtransform
Mtmatrix dtransform
} bind def
/Mabsadd { Mabs
3 -1 roll add
3 1 roll add
exch } bind def
%% Graphics
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10  scalefont  setfont
% Scaling calculations
0.0238095 5.29101 0 0.561849 [
[.02381 -0.0125 -3 -9 ]
[.02381 -0.0125 3 0 ]
[.15608 -0.0125 -15 -9 ]
[.15608 -0.0125 15 0 ]
[.28836 -0.0125 -12 -9 ]
[.28836 -0.0125 12 0 ]
[.42063 -0.0125 -15 -9 ]
[.42063 -0.0125 15 0 ]
[.55291 -0.0125 -9 -9 ]
[.55291 -0.0125 9 0 ]
[.68519 -0.0125 -15 -9 ]
[.68519 -0.0125 15 0 ]
[.81746 -0.0125 -12 -9 ]
[.81746 -0.0125 12 0 ]
[.94974 -0.0125 -15 -9 ]
[.94974 -0.0125 15 0 ]
[ 0 0 -0.125 0 ]
[.5 -0.0125 -35 -21 ]
[.5 -0.0125 35 -9 ]
[-0.0125 .11237 -18 -4.5 ]
[-0.0125 .11237 0 4.5 ]
[-0.0125 .22474 -18 -4.5 ]
[-0.0125 .22474 0 4.5 ]
[-0.0125 .33711 -18 -4.5 ]
[-0.0125 .33711 0 4.5 ]
[-0.0125 .44948 -18 -4.5 ]
[-0.0125 .44948 0 4.5 ]
[-0.0125 .56185 -6 -4.5 ]
[-0.0125 .56185 0 4.5 ]
[ 0 0 -0.125 0 ]
[-0.01875 .30902 -30 -50 ]
[-0.01875 .30902 -18 50 ]
[ 0 .61803 .125 0 ]
[ 1 0 .125 0 ]
[ 0 0 0 0 ]
[ 1 .61803 0 0 ]
] MathScale
% Start of Graphics
1 setlinecap
1 setlinejoin
newpath
0 g
.25 Mabswid
[ ] 0 setdash
.02381 0 m
.02381 .00625 L
s
[(0)] .02381 -0.0125 0 1 Mshowa
.15608 0 m
.15608 .00625 L
s
[(0.025)] .15608 -0.0125 0 1 Mshowa
.28836 0 m
.28836 .00625 L
s
[(0.05)] .28836 -0.0125 0 1 Mshowa
.42063 0 m
.42063 .00625 L
s
[(0.075)] .42063 -0.0125 0 1 Mshowa
.55291 0 m
.55291 .00625 L
s
[(0.1)] .55291 -0.0125 0 1 Mshowa
.68519 0 m
.68519 .00625 L
s
[(0.125)] .68519 -0.0125 0 1 Mshowa
.81746 0 m
.81746 .00625 L
s
[(0.15)] .81746 -0.0125 0 1 Mshowa
.94974 0 m
.94974 .00625 L
s
[(0.175)] .94974 -0.0125 0 1 Mshowa
.125 Mabswid
.05026 0 m
.05026 .00375 L
s
.07672 0 m
.07672 .00375 L
s
.10317 0 m
.10317 .00375 L
s
.12963 0 m
.12963 .00375 L
s
.18254 0 m
.18254 .00375 L
s
.20899 0 m
.20899 .00375 L
s
.23545 0 m
.23545 .00375 L
s
.2619 0 m
.2619 .00375 L
s
.31481 0 m
.31481 .00375 L
s
.34127 0 m
.34127 .00375 L
s
.36772 0 m
.36772 .00375 L
s
.39418 0 m
.39418 .00375 L
s
.44709 0 m
.44709 .00375 L
s
.47354 0 m
.47354 .00375 L
s
.5 0 m
.5 .00375 L
s
.52646 0 m
.52646 .00375 L
s
.57937 0 m
.57937 .00375 L
s
.60582 0 m
.60582 .00375 L
s
.63228 0 m
.63228 .00375 L
s
.65873 0 m
.65873 .00375 L
s
.71164 0 m
.71164 .00375 L
s
.7381 0 m
.7381 .00375 L
s
.76455 0 m
.76455 .00375 L
s
.79101 0 m
.79101 .00375 L
s
.84392 0 m
.84392 .00375 L
s
.87037 0 m
.87037 .00375 L
s
.89683 0 m
.89683 .00375 L
s
.92328 0 m
.92328 .00375 L
s
.97619 0 m
.97619 .00375 L
s
.25 Mabswid
0 0 m
1 0 L
s
gsave
.5 -0.0125 -96 -25 Mabsadd m
1 1 Mabs scale
currentpoint translate
0 20 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.000000 moveto
 558.000000 16.000000 lineto
 558.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.000000 lineto
closepath
clip newpath
63.000000 13.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 13.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(stdDevError) show
129.000000 13.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
0 .11237 m
.00625 .11237 L
s
[(0.2)] -0.0125 .11237 1 0 Mshowa
0 .22474 m
.00625 .22474 L
s
[(0.4)] -0.0125 .22474 1 0 Mshowa
0 .33711 m
.00625 .33711 L
s
[(0.6)] -0.0125 .33711 1 0 Mshowa
0 .44948 m
.00625 .44948 L
s
[(0.8)] -0.0125 .44948 1 0 Mshowa
0 .56185 m
.00625 .56185 L
s
[(1)] -0.0125 .56185 1 0 Mshowa
.125 Mabswid
0 .02809 m
.00375 .02809 L
s
0 .05618 m
.00375 .05618 L
s
0 .08428 m
.00375 .08428 L
s
0 .14046 m
.00375 .14046 L
s
0 .16855 m
.00375 .16855 L
s
0 .19665 m
.00375 .19665 L
s
0 .25283 m
.00375 .25283 L
s
0 .28092 m
.00375 .28092 L
s
0 .30902 m
.00375 .30902 L
s
0 .3652 m
.00375 .3652 L
s
0 .39329 m
.00375 .39329 L
s
0 .42139 m
.00375 .42139 L
s
0 .47757 m
.00375 .47757 L
s
0 .50566 m
.00375 .50566 L
s
0 .53376 m
.00375 .53376 L
s
0 .58994 m
.00375 .58994 L
s
.25 Mabswid
0 0 m
0 .61803 L
s
gsave
-0.01875 .30902 -135 -10 Mabsadd m
1 1 Mabs scale
currentpoint translate
111 10 translate
90 rotate
-111 -10 translate
0 20 translate 1 -1 scale
gsave
0.000000 0.000000 0.000000 setrgbcolor
1.000000 setlinewidth
gsave
newpath
 61.000000 16.000000 moveto
 558.000000 16.000000 lineto
 558.000000 4.000000 lineto
 61.000000 4.000000 lineto
 61.000000 16.000000 lineto
closepath
clip newpath
63.000000 13.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
63.000000 13.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
(fraction) show
117.000000 13.000000 moveto
(correct) show
159.000000 13.000000 moveto
%%IncludeResource: font Courier
%%IncludeFont: Courier
/Courier findfont 10.000000 scalefont
[1 0 0 -1 0 0 ] makefont setfont
0.000000 0.000000 0.000000 setrgbcolor
0.000000 0.000000 rmoveto
1.000000 setlinewidth
grestore
grestore
%%DocumentNeededResources: font Courier
%%DocumentSuppliedResources:
%%DocumentNeededFonts: Courier
%%DocumentSuppliedFonts:
%%DocumentFonts: font Courier
grestore
.02381 .61178 m
.02381 .61803 L
s
.15608 .61178 m
.15608 .61803 L
s
.28836 .61178 m
.28836 .61803 L
s
.42063 .61178 m
.42063 .61803 L
s
.55291 .61178 m
.55291 .61803 L
s
.68519 .61178 m
.68519 .61803 L
s
.81746 .61178 m
.81746 .61803 L
s
.94974 .61178 m
.94974 .61803 L
s
.125 Mabswid
.05026 .61428 m
.05026 .61803 L
s
.07672 .61428 m
.07672 .61803 L
s
.10317 .61428 m
.10317 .61803 L
s
.12963 .61428 m
.12963 .61803 L
s
.18254 .61428 m
.18254 .61803 L
s
.20899 .61428 m
.20899 .61803 L
s
.23545 .61428 m
.23545 .61803 L
s
.2619 .61428 m
.2619 .61803 L
s
.31481 .61428 m
.31481 .61803 L
s
.34127 .61428 m
.34127 .61803 L
s
.36772 .61428 m
.36772 .61803 L
s
.39418 .61428 m
.39418 .61803 L
s
.44709 .61428 m
.44709 .61803 L
s
.47354 .61428 m
.47354 .61803 L
s
.5 .61428 m
.5 .61803 L
s
.52646 .61428 m
.52646 .61803 L
s
.57937 .61428 m
.57937 .61803 L
s
.60582 .61428 m
.60582 .61803 L
s
.63228 .61428 m
.63228 .61803 L
s
.65873 .61428 m
.65873 .61803 L
s
.71164 .61428 m
.71164 .61803 L
s
.7381 .61428 m
.7381 .61803 L
s
.76455 .61428 m
.76455 .61803 L
s
.79101 .61428 m
.79101 .61803 L
s
.84392 .61428 m
.84392 .61803 L
s
.87037 .61428 m
.87037 .61803 L
s
.89683 .61428 m
.89683 .61803 L
s
.92328 .61428 m
.92328 .61803 L
s
.97619 .61428 m
.97619 .61803 L
s
.25 Mabswid
0 .61803 m
1 .61803 L
s
.99375 0 m
1 0 L
s
.99375 .11237 m
1 .11237 L
s
.99375 .22474 m
1 .22474 L
s
.99375 .33711 m
1 .33711 L
s
.99375 .44948 m
1 .44948 L
s
.99375 .56185 m
1 .56185 L
s
.125 Mabswid
.99625 .02809 m
1 .02809 L
s
.99625 .05618 m
1 .05618 L
s
.99625 .08428 m
1 .08428 L
s
.99625 .14046 m
1 .14046 L
s
.99625 .16855 m
1 .16855 L
s
.99625 .19665 m
1 .19665 L
s
.99625 .25283 m
1 .25283 L
s
.99625 .28092 m
1 .28092 L
s
.99625 .30902 m
1 .30902 L
s
.99625 .3652 m
1 .3652 L
s
.99625 .39329 m
1 .39329 L
s
.99625 .42139 m
1 .42139 L
s
.99625 .47757 m
1 .47757 L
s
.99625 .50566 m
1 .50566 L
s
.99625 .53376 m
1 .53376 L
s
.99625 .58994 m
1 .58994 L
s
.25 Mabswid
1 0 m
1 .61803 L
s
0 0 m
1 0 L
1 .61803 L
0 .61803 L
closepath
clip
newpath
.002 w
.02381 .56185 m
.02381 .56185 L
s
.015 w
.02381 .56185 Mdot
.002 w
.18254 .56185 m
.18254 .56185 L
s
.015 w
.18254 .56185 Mdot
.002 w
.34127 .08228 m
.34127 .47957 L
s
.015 w
.34127 .28092 Mdot
.002 w
.5 .34897 m
.5 .54999 L
s
.015 w
.5 .44948 Mdot
.002 w
.65873 .14046 m
.65873 .42139 L
s
.015 w
.65873 .28092 Mdot
.002 w
.81746 .08228 m
.81746 .47957 L
s
.015 w
.81746 .28092 Mdot
.002 w
.97619 .22165 m
.97619 .52748 L
s
.015 w
.97619 .37457 Mdot
% End of Graphics
MathPictureEnd
\
\>"], "Graphics",
  ImageSize->{288, 177.938},
  ImageMargins->{{43, 0}, {0, 0}},
  ImageRegion->{{0, 1}, {0, 1}},
  ImageCache->GraphicsData["Bitmap", "\<\
CF5dJ6E]HGAYHf4PAg9QL6QYHg<PAVmbKF5d0`40004P0000/A000`40O003h00Oogoo8Goo003oOolQ
Ool00?moob5oo`00ogoo8Goo001oOol40003Ool30003Ool90003Ool40002Ool20002Ool50002Ool4
0002Ool40003Ool30002Ool4001QOol0089oo`05001oogooOol00002Ool00`00Ool00003Ool00`00
Ool00003Ool00`00Ool00006Ool20003Ool00`00Oomoo`04Ool00`00Oomoo`03Ool00`00Oomoo`02
Ool01@00Oomoogoo00000Woo00<007ooOol0HGoo001oOol40003Ool00`00Oomoo`02Ool01`00Oomo
ogoo001oo`0000=oo`03001oo`0000@00003Ool007oo0080009oo`03001oogoo00Aoo`03001oogoo
00=oo`03001oogoo009oo`05001oogooOol00002Ool00`00Oomoo`1QOol007moo`03001oogoo00Ao
o`03001oogoo009oo`07001oogooOol007oo00000goo00<007oo00000goo00<007oo00000Woo00<0
07ooOol00`0017oo0P0017oo0P000goo00D007ooOomoo`00009oo`80069oo`00Ogoo10000Woo1000
0goo100000=oo`00Ool00Woo00<007ooOol00`0000=oo`0000000Woo0P0000=oo`00Ool017oo0P00
00Aoo`00001oo`800003Ool00000009oo`<0009oo`800003Ool00000061oo`00QWoo00<007ooOol0
1Woo1P003Goo1@00NGoo0026Ool00`00Oomoo`05Ool2002@Ool00?moob5oo`00ogoo8Goo003oOolQ
Ool002Qoo`8001=oo`8000Aoo`8000Aoo`8000=oo`@0009oo`<000Yoo`8000Aoo`8000Aoo`8000=o
o`<000Uoo`8000Aoo`8000Aoo`8000Aoo`03001oogoo009oo`<000eoo`8000Aoo`8000=oo`<000eo
o`8000Aoo`8000=oo`<000=oo`@0009oo`<000Yoo`8000Aoo`8000=oo`<000=oo`<000Uoo`8000Ao
o`8000=oo`<000Aoo`03001oogoo009oo`<000Eoo`009goo00@007ooOol0015oo`04001oogoo0003
Ool20003Ool01000Oomoo`000goo00<007ooOol01Goo00<007ooOol01Woo00@007ooOol000=oo`80
00=oo`04001oogoo0005Ool00`00Oomoo`05Ool01000Oomoo`000goo0P000goo00@007ooOol000=o
o`03001oogoo00Eoo`03001oogoo00Uoo`04001oogoo0003Ool20004Ool00`00Oomoo`0;Ool01000
Oomoo`000goo0P0017oo00<007ooOol00goo00<007ooOol01Goo00<007ooOol01Woo00@007ooOol0
00=oo`8000Aoo`03001oogoo00Eoo`03001oogoo00Eoo`04001oogoo0003Ool20004Ool00`00Oomo
o`03Ool00`00Oomoo`05Ool00`00Oomoo`02Ool002Moo`04001oogoo000AOol01000Oomoo`0027oo
00@007ooOol000Aoo`03001oogoo00Aoo`03001oogoo00Ioo`04001oogoo0008Ool01000Oomoo`00
1Goo00<007ooOol01Goo00@007ooOol000Qoo`04001oogoo0004Ool00`00Oomoo`04Ool00`00Oomo
o`09Ool01000Oomoo`002Goo00<007ooOol02goo00@007ooOol000Uoo`03001oogoo00Aoo`03001o
ogoo00Aoo`03001oogoo00Ioo`04001oogoo0009Ool00`00Oomoo`05Ool00`00Oomoo`05Ool01000
Oomoo`002Goo00<007ooOol017oo00<007ooOol017oo00<007ooOol00Woo000WOol01000Oomoo`00
4Goo00@007ooOol000Qoo`04001oogoo0005Ool00`00Oomoo`030009Ool01000Oomoo`0027oo00@0
07ooOol0009oo`<000Qoo`04001oogoo0008Ool01000Oomoo`0017oo00@007ooOomoo`<000aoo`04
001oogoo0009Ool00`00Oomoo`0;Ool01000Oomoo`002Goo00<007ooOol01Goo00<007ooOol00`00
2Goo00@007ooOol000Uoo`03001oogoo009oo`<000Qoo`04001oogoo0009Ool00`00Oomoo`04Ool0
1000Oomoogoo0`001Goo000WOol01000Oomoo`004Goo00@007ooOol000Qoo`04001oogoo0002Ool0
1000Oomoo`000Woo00<007ooOol02Goo00@007ooOol000Qoo`04001oogoo0002Ool00`00Oomoo`08
Ool01000Oomoo`0027oo00@007ooOol000Eoo`04001oogoo000>Ool01000Oomoo`002Goo00<007oo
Ool02goo00@007ooOol000Uoo`03001oogoo009oo`04001oogoo0002Ool00`00Oomoo`09Ool01000
Oomoo`002Goo00<007ooOol00Woo00<007ooOol027oo00@007ooOol000Uoo`03001oogoo00Eoo`04
001oogoo0007Ool002Qoo`8001=oo`8000Yoo`8000Aoo`8000=oo`@000Uoo`8000Yoo`8000=oo`@0
00Qoo`8000Yoo`8000=oo`@0009oo`@000aoo`8000Uoo`8000ioo`8000Uoo`8000Eoo`8000=oo`@0
00Uoo`8000Uoo`8000Aoo`@000Qoo`8000Uoo`8000Aoo`@0009oo`@000Aoo`00ogoo8Goo003oOolQ
Ool00?moob5oo`00ogoo8Goo000SOooi0004Ool002=oo`03001oogoo00=oo`03001oogoo00Aoo`03
001oogoo00=oo`03001oogoo00Aoo`03001oogoo00=oo`03001oogoo00Aoo`03001oogoo00Aoo`03
001oogoo00=oo`03001oogoo00Aoo`03001oogoo00=oo`03001oogoo00Aoo`03001oogoo00=oo`03
001oogoo00Aoo`03001oogoo00=oo`03001oogoo00Aoo`03001oogoo00=oo`03001oogoo00Aoo`03
001oogoo00=oo`03001oogoo00Aoo`03001oogoo00Aoo`03001oogoo00=oo`03001oogoo00Aoo`03
001oogoo00=oo`03001oogoo00Aoo`03001oogoo00=oo`03001oogoo00Aoo`03001oogoo00=oo`03
001oogoo00Aoo`03001oogoo00=oo`03001oogoo00Aoo`03001oogoo00Aoo`03001oogoo00=oo`03
001oogoo00Aoo`03001oogoo00=oo`03001oogoo00Aoo`03001oogoo00=oo`03001oogoo00Aoo`03
001oogoo00=oo`03001oogoo009oo`008goo00<007ooOol0mGoo00<007ooOol00Woo000SOol00`00
Oomoo`3eOol00`00Oomoo`02Ool002=oo`03001oogoo0?Eoo`03001oogoo009oo`008goo00<007oo
Ool0mGoo00<007ooOol00Woo000SOol00`00Oomoo`3eOol00`00Oomoo`02Ool002=oo`800?Eoo`80
00Aoo`008goo00<007ooOol0mGoo00<007ooOol00Woo000SOol00`00Oomoo`3eOol00`00Oomoo`02
Ool002=oo`03001oogoo0?Eoo`03001oogoo009oo`008goo00<007ooOol0mGoo00<007ooOol00Woo
000SOol00`00Oomoo`3eOol00`00Oomoo`02Ool002=oo`03001oogoo0?Eoo`03001oogoo009oo`00
8goo0P00mGoo0P0017oo000SOol00`00Oomoo`3eOol00`00Oomoo`02Ool002=oo`03001oogoo0?Eo
o`03001oogoo009oo`008goo00<007ooOol0mGoo00<007ooOol00Woo000SOol00`00Oomoo`3eOol0
0`00Oomoo`02Ool002=oo`03001oogoo0?Eoo`03001oogoo009oo`008goo0P00Dgoo00<007ooOol0
Lgoo00<007ooOol0:Goo0P0017oo000SOol00`00Oomoo`1BOol00`00Oomoo`1cOol00`00Oomoo`0Z
Ool00`00Oomoo`02Ool002=oo`03001oogoo059oo`03001oogoo07=oo`03001oogoo02Yoo`03001o
ogoo009oo`008goo00<007ooOol0DWoo00<007ooOol0Lgoo00<007ooOol0:Woo00<007ooOol00Woo
000SOol00`00Oomoo`1BOol00`00Oomoo`1cOol00`00Oomoo`0ZOol00`00Oomoo`02Ool0011oo`80
00Aoo`8000=oo`@000Aoo`03001oogoo059oo`03001oogoo07=oo`03001oogoo02Yoo`03001oogoo
009oo`003goo00@007ooOol000=oo`8000Aoo`03001oogoo00Aoo`03001oogoo059oo`03001oogoo
07=oo`03001oogoo02Yoo`03001oogoo009oo`003goo00@007ooOol000Yoo`03001oogoo00=oo`<0
059oo`03001oogoo07=oo`03001oogoo02Qoo`<000Aoo`003goo00@007ooOol000]oo`03001oogoo
009oo`03001oogoo059oo`03001oogoo07=oo`03001oogoo02Yoo`03001oogoo009oo`002goo1@00
0Woo00<007ooOol01Woo00@007ooOol000Aoo`03001oogoo059oo`03001oogoo07=oo`03001oogoo
02Yoo`03001oogoo009oo`0037oo00@007ooOomoo`8000Yoo`8000Eoo`03001oogoo059oo`03001o
ogoo07=oo`03001oogoo02Yoo`03001oogoo009oo`0037oo00<007ooOol057oo00<007ooOol0DWoo
00<007ooOol0Lgoo00<007ooOol0:Woo00<007ooOol00Woo000<Ool00`00Oomoo`0DOol00`00Oomo
o`1BOol00`00Oomoo`1cOol00`00Oomoo`0ZOol00`00Oomoo`02Ool000]oo`@001Aoo`03001oogoo
059oo`03001oogoo07=oo`03001oogoo02Yoo`03001oogoo009oo`0037oo00<007ooOol057oo0P00
Dgoo00<007ooOol0Bgoo00<007ooOol09Goo00<007ooOol0:Goo0P0017oo000;Ool4000DOol00`00
Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0ZOol00`00Oomoo`02Ool000ao
o`03001oogoo01Aoo`03001oogoo059oo`03001oogoo04]oo`03001oogoo02Eoo`03001oogoo02Yo
o`03001oogoo009oo`0037oo00<007ooOol057oo00<007ooOol0DWoo00<007ooOol0Bgoo00<007oo
Ool09Goo00<007ooOol0:Woo00<007ooOol00Woo000<Ool2000EOol00`00Oomoo`1BOol00`00Oomo
o`1;Ool00`00Oomoo`0UOol00`00Oomoo`0ZOol00`00Oomoo`02Ool000]oo`800003Ool0000001=o
o`03001oogoo059oo`03001oogoo04]oo`03001oogoo02Eoo`03001oogoo02Yoo`03001oogoo009o
o`008goo00<007ooOol0DWoo00<007ooOol0Bgoo00<007ooOol09Goo00<007ooOol0:Woo00<007oo
Ool00Woo000;Ool5000COol2001COol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0YOol2
0004Ool000]oo`04001oogoo000DOol00`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol0
0`00Oomoo`0ZOol00`00Oomoo`02Ool000]oo`@001Aoo`03001oogoo059oo`03001oogoo04]oo`03
001oogoo02Eoo`03001oogoo02Yoo`03001oogoo009oo`003Woo00<007ooOol04Woo00<007ooOol0
DWoo00<007ooOol0Bgoo00<007ooOol09Goo00<007ooOol0:Woo00<007ooOol00Woo000<Ool3000D
Ool00`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0ZOol00`00Oomoo`02
Ool002=oo`03001oogoo059oo`03001oogoo04]oo`03001oogoo02Eoo`03001oogoo02Yoo`03001o
ogoo009oo`0037oo0`0057oo00<007ooOol0DWoo00<007ooOol0Bgoo00<007ooOol09Goo00<007oo
Ool0:Woo00<007ooOol00Woo000;Ool01@00Oomoogoo00004goo0P00Dgoo00<007ooOol0Bgoo00<0
07ooOol09Goo00<007ooOol0:Goo0P0017oo000;Ool00`00Oomoo`0EOol00`00Oomoo`1BOol00`00
Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0ZOol00`00Oomoo`02Ool000]oo`05001oogooOol0
000COol00`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0ZOol00`00Oomo
o`02Ool000aoo`@001=oo`03001oogoo059oo`03001oogoo04]oo`03001oogoo02Eoo`03001oogoo
02Yoo`03001oogoo009oo`008goo00<007ooOol0DWoo00<007ooOol0Bgoo00<007ooOol09Goo00<0
07ooOol0:Woo00<007ooOol00Woo000<Ool300000goo00000004Ool20004Ool30004Ool00`00Oomo
o`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0ZOol00`00Oomoo`02Ool000aoo`04
001oogoo0002Ool01000Oomoogoo0P001Goo00<007ooOol00goo00<007ooOol0DWoo00<007ooOol0
Bgoo00<007ooOol09Goo00<007ooOol097oo00<007ooOol00goo00<007ooOol00Woo000<Ool01000
Oomoo`000Woo00<007ooOol01Woo100017oo0`00DWoo00<007ooOol0Bgoo00<007ooOol09Goo00<0
07ooOol097oo00@007ooOomoo`<000Aoo`0037oo00@007ooOol0009oo`03001oogoo00Ioo`03001o
o`0000Eoo`03001oogoo059oo`03001oogoo04]oo`03001oogoo02Eoo`03001oogoo02Aoo`03001o
ogoo00=oo`03001oogoo009oo`002goo1@000Woo00<007ooOol01Woo00<007oo00001Goo00<007oo
Ool0DWoo00<007ooOol0Bgoo00<007ooOol09Goo00<007ooOol097oo00<007ooOol00goo00<007oo
Ool00Woo000<Ool01000Oomoogoo0P002Woo0P001Goo00<007ooOol0DWoo00<007ooOol0Bgoo00<0
07ooOol09Goo00<007ooOol097oo00<007ooOol00goo00<007ooOol00Woo000;Ool5000COol00`00
Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0TOol00`00Oomoo`03Ool00`00
Oomoo`02Ool000eoo`03001oogoo01=oo`03001oogoo059oo`03001oogoo04]oo`03001oogoo02Eo
o`03001oogoo02Aoo`03001oogoo00=oo`03001oogoo009oo`003Goo00<007ooOol04goo00<007oo
Ool0DWoo00<007ooOol0Bgoo00<007ooOol09Goo00<007ooOol097oo00<007ooOol00goo00<007oo
Ool00Woo000=Ool00`00Oomoo`0COol2001COol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomo
o`0TOol00`00Oomoo`02Ool20004Ool000aoo`8001Eoo`03001oogoo059oo`03001oogoo04]oo`03
001oogoo02Eoo`03001oogoo02Aoo`03001oogoo00=oo`03001oogoo009oo`008goo00<007ooOol0
DWoo00<007ooOol0Bgoo00<007ooOol09Goo00<007ooOol097oo00<007ooOol00goo00<007ooOol0
0Woo000<Ool3000DOol00`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0T
Ool00`00Oomoo`03Ool00`00Oomoo`02Ool000]oo`05001oogooOol0000COol00`00Oomoo`1BOol0
0`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0TOol00`00Oomoo`03Ool00`00Oomoo`02Ool0
00]oo`05001oogooOol0000COol00`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00
Oomoo`0TOol00`00Oomoo`03Ool00`00Oomoo`02Ool000]oo`05001oogooOol0000COol00`00Oomo
o`1@Ool3001<Ool3000TOol3000VOol00`00Oomoo`03Ool00`00Oomoo`02Ool000aoo`<001Aoo`80
055oo`<004aoo`<002Aoo`<002Ioo`03001oogoo009oo`8000Aoo`008goo00<007ooOol0D7oo0`00
C7oo0`0097oo0`009Woo00<007ooOol00goo00<007ooOol00Woo000;Ool200000goo0000000COol0
0`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0TOol00`00Oomoo`03Ool0
0`00Oomoo`02Ool000]oo`04001oogoo000DOol00`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomo
o`0UOol00`00Oomoo`0TOol00`00Oomoo`03Ool00`00Oomoo`02Ool000]oo`04001oogoo000DOol0
0`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0TOol00`00Oomoo`03Ool0
0`00Oomoo`02Ool000]oo`04001oogoo000DOol00`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomo
o`0UOol00`00Oomoo`0TOol00`00Oomoo`03Ool00`00Oomoo`02Ool000Yoo`D001Aoo`03001oogoo
059oo`03001oogoo04]oo`03001oogoo02Eoo`03001oogoo02Aoo`03001oogoo00=oo`03001oogoo
009oo`008goo0P00Dgoo00<007ooOol0Bgoo00<007ooOol09Goo00<007ooOol097oo00<007ooOol0
0Woo0P0017oo000SOol00`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0T
Ool00`00Oomoo`03Ool00`00Oomoo`02Ool002=oo`03001oogoo059oo`03001oogoo04]oo`03001o
ogoo02Eoo`03001oogoo02Aoo`03001oogoo00=oo`03001oogoo009oo`008goo00<007ooOol0DWoo
00<007ooOol0Bgoo00<007ooOol09Goo00<007ooOol097oo00<007ooOol00goo00<007ooOol00Woo
000SOol00`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0TOol00`00Oomo
o`03Ool00`00Oomoo`02Ool0011oo`8000Aoo`8000Aoo`8000Eoo`03001oogoo059oo`03001oogoo
04]oo`03001oogoo02Eoo`03001oogoo02Aoo`03001oogoo00=oo`03001oogoo009oo`003goo00@0
07ooOol000=oo`8000=oo`04001oogoo0004Ool00`00Oomoo`1BOol00`00Oomoo`1;Ool00`00Oomo
o`0UOol00`00Oomoo`0TOol00`00Oomoo`03Ool00`00Oomoo`02Ool000aoo`@0009oo`03001oogoo
00Ioo`04001oogoo0004Ool3001BOol00`00Oomoo`1;Ool00`00Oomoo`0UOol00`00Oomoo`0TOol0
1000Oomoogoo0`0017oo000;Ool01@00Oomoogoo00000Woo00<007ooOol01Woo0`001Goo00<007oo
Ool0DWoo00<007ooOol0Bgoo00<007ooOol09Goo00<007ooOol097oo00<007ooOol00goo00<007oo
Ool00Woo000;Ool01@00Oomoogoo00000Woo00<007ooOol01Woo00<007ooOol01Goo00<007ooOol0
DWoo00<007ooOol0Bgoo00<007ooOol09Goo00<007ooOol097oo00<007ooOol00goo00<007ooOol0
0Woo000;Ool01000Oomoogoo0`002Woo0`0017oo00<007ooOol0DWoo00<007ooOol097oo00<007oo
Ool097oo00<007ooOol09Goo00<007ooOol097oo00<007ooOol00goo00<007ooOol00Woo000<Ool4
000COol00`00Oomoo`1BOol00`00Oomoo`0TOol00`00Oomoo`0TOol00`00Oomoo`0UOol00`00Oomo
o`0TOol00`00Oomoo`03Ool00`00Oomoo`02Ool002=oo`03001oogoo059oo`03001oogoo02Aoo`03
001oogoo02Aoo`03001oogoo02Eoo`03001oogoo02Aoo`03001oogoo00=oo`03001oogoo009oo`00
37oo0`0057oo00<007ooOol0DWoo00<007ooOol097oo00<007ooOol097oo00<007ooOol09Goo00<0
07ooOol097oo00<007ooOol00goo00<007ooOol00Woo000;Ool01@00Oomoogoo00004goo0P00Dgoo
00<007ooOol097oo00<007ooOol097oo00<007ooOol09Goo00<007ooOol097oo00<007ooOol00Woo
0P0017oo000;Ool01@00Oomoogoo00004goo00<007ooOol0DWoo00<007ooOol097oo00<007ooOol0
97oo00<007ooOol09Goo00<007ooOol097oo00<007ooOol00goo00<007ooOol00Woo000;Ool01@00
Oomoogoo00004goo00<007ooOol0DWoo00<007ooOol097oo00<007ooOol097oo00<007ooOol09Goo
00<007ooOol08Woo0`001Goo00<007ooOol00Woo000<Ool3000DOol00`00Oomoo`1BOol00`00Oomo
o`0TOol00`00Oomoo`0TOol00`00Oomoo`0UOol00`00Oomoo`0ROol30005Ool00`00Oomoo`02Ool0
02=oo`03001oogoo059oo`03001oogoo02Aoo`03001oogoo02Aoo`03001oogoo02Eoo`03001oogoo
029oo`<000Eoo`03001oogoo009oo`002goo100057oo00<007ooOol0DWoo00<007ooOol097oo00<0
07ooOol097oo00<007ooOol09Goo00<007ooOol097oo00<007ooOol00goo00<007ooOol00Woo000<
Ool00`00Oomoo`0DOol00`00Oomoo`1BOol00`00Oomoo`0TOol00`00Oomoo`0TOol00`00Oomoo`0U
Ool00`00Oomoo`0TOol00`00Oomoo`03Ool00`00Oomoo`02Ool000aoo`03001oogoo01Aoo`8005=o
o`03001oogoo02Aoo`03001oogoo02Aoo`03001oogoo02Eoo`03001oogoo02Aoo`03001oogoo009o
o`8000Aoo`0037oo0P005Goo00<007ooOol0DWoo00<007ooOol097oo00<007ooOol097oo00<007oo
Ool09Goo00<007ooOol097oo00<007ooOol00goo00<007ooOol00Woo000;Ool200000goo0000000C
Ool00`00Oomoo`1BOol00`00Oomoo`0TOol00`00Oomoo`0TOol00`00Oomoo`0UOol00`00Oomoo`0T
Ool00`00Oomoo`03Ool00`00Oomoo`02Ool002=oo`03001oogoo059oo`03001oogoo02Aoo`03001o
ogoo02Aoo`03001oogoo02Eoo`03001oogoo02Aoo`03001oogoo00=oo`03001oogoo009oo`002goo
100057oo00<007ooOol0DWoo00<007ooOol097oo00<007ooOol097oo00<007ooOol09Goo00<007oo
Ool097oo00<007ooOol00goo00<007ooOol00Woo000<Ool00`00Oomoo`0DOol00`00Oomoo`1BOol0
0`00Oomoo`0TOol00`00Oomoo`0TOol00`00Oomoo`0UOol00`00Oomoo`0TOol00`00Oomoo`03Ool0
0`00Oomoo`02Ool000aoo`03001oogoo01Aoo`03001oogoo059oo`03001oogoo02Aoo`03001oogoo
02Aoo`03001oogoo02Eoo`03001oogoo02Aoo`03001oogoo00=oo`03001oogoo009oo`0037oo0P00
5Goo0P00Dgoo00<007ooOol097oo00<007ooOol097oo00<007ooOol09Goo00<007ooOol097oo00<0
07ooOol00Woo0P0017oo000;Ool200000goo0000000COol00`00Oomoo`1BOol00`00Oomoo`0TOol0
0`00Oomoo`1<Ool00`00Oomoo`0TOol00`00Oomoo`03Ool00`00Oomoo`02Ool002=oo`03001oogoo
059oo`03001oogoo02Aoo`03001oogoo04aoo`03001oogoo02Aoo`03001oogoo00=oo`03001oogoo
009oo`0037oo10004goo00<007ooOol0DWoo00<007ooOol097oo00<007ooOol0C7oo00<007ooOol0
97oo00<007ooOol00goo00<007ooOol00Woo000;Ool00`00Oomoo`0EOol00`00Oomoo`1BOol00`00
Oomoo`0TOol00`00Oomoo`1<Ool00`00Oomoo`0TOol00`00Oomoo`03Ool00`00Oomoo`02Ool000]o
o`L000Aoo`8000Aoo`8000Eoo`03001oogoo059oo`03001oogoo02Aoo`03001oogoo04aoo`03001o
ogoo02Aoo`03001oogoo00=oo`03001oogoo009oo`002goo00D007ooOomoo`00009oo`04001oogoo
Ool20003Ool01000Oomoo`0017oo00<007ooOol0DWoo00<007ooOol08goo0`00CGoo00<007ooOol0
97oo00<007ooOol00goo00<007ooOol00Woo000<Ool40002Ool00`00Oomoo`06Ool01000Oomoo`00
17oo0`00DWoo00<007ooOol08goo0`00CGoo00<007ooOol097oo00@007ooOomoo`<000Aoo`003goo
00@007ooOol000Uoo`8000Eoo`03001oogoo059oo`03001oogoo02=oo`<004eoo`03001oogoo02Ao
o`03001oogoo00=oo`03001oogoo009oo`0037oo10000Woo00<007ooOol01Woo00@007ooOol000Ao
o`03001oogoo059oo`03001oogoo02Aoo`03001oogoo04aoo`03001oogoo02Aoo`03001oogoo00=o
o`03001oogoo009oo`002goo00@007ooOomoo`<000Yoo`8000Eoo`03001oogoo059oo`03001oogoo
02Aoo`03001oogoo04aoo`03001oogoo02Aoo`03001oogoo00=oo`03001oogoo009oo`002goo00<0
07ooOol05Goo00<007ooOol0DWoo00<007ooOol097oo00<007ooOol0C7oo00<007ooOol097oo00<0
07ooOol00goo00<007ooOol00Woo000;Ool01@00Oomoogoo00004goo00<007ooOol0DWoo00<007oo
Ool097oo00<007ooOol0C7oo00<007ooOol097oo00<007ooOol00goo00<007ooOol00Woo000<Ool4
000COol00`00Oomoo`1BOol00`00Oomoo`0TOol00`00Oomoo`1<Ool00`00Oomoo`0TOol00`00Oomo
o`03Ool00`00Oomoo`02Ool002=oo`8005=oo`03001oogoo02Aoo`03001oogoo04aoo`03001oogoo
02Aoo`03001oogoo009oo`8000Aoo`0037oo0`0057oo00<007ooOol0NGoo00<007ooOol0Lgoo00<0
07ooOol00goo00<007ooOol00Woo000<Ool01000Oomoo`004goo00<007ooOol0NGoo00<007ooOol0
Lgoo00<007ooOol00goo00<007ooOol00Woo000<Ool00`00Oomoo`0DOol00`00Oomoo`1iOol00`00
Oomoo`1cOol00`00Oomoo`03Ool00`00Oomoo`02Ool000aoo`03001oogoo01Aoo`03001oogoo07Uo
o`03001oogoo07=oo`03001oogoo00=oo`03001oogoo009oo`002goo100057oo00<007ooOol0NGoo
00<007ooOol0Lgoo00<007ooOol00goo00<007ooOol00Woo000<Ool00`00Oomoo`0DOol00`00Oomo
o`1iOol00`00Oomoo`1cOol00`00Oomoo`03Ool00`00Oomoo`02Ool000aoo`03001oogoo01Aoo`80
07Yoo`03001oogoo07=oo`03001oogoo009oo`8000Aoo`008goo00<007ooOol0NGoo00<007ooOol0
Lgoo00<007ooOol00goo00<007ooOol00Woo000SOol00`00Oomoo`1iOol00`00Oomoo`1cOol00`00
Oomoo`03Ool00`00Oomoo`02Ool002=oo`03001oogoo07Uoo`03001oogoo07=oo`03001oogoo00=o
o`03001oogoo009oo`008goo00<007ooOol0NGoo00<007ooOol0Lgoo00<007ooOol00goo00<007oo
Ool00Woo000SOol00`00Oomoo`1iOol00`00Oomoo`1cOol00`00Oomoo`03Ool00`00Oomoo`02Ool0
02=oo`03001oogoo07Uoo`03001oogoo07Uoo`03001oogoo009oo`008goo0P00NWoo00<007ooOol0
N7oo0P0017oo000SOol00`00Oomoo`1iOol00`00Oomoo`1iOol00`00Oomoo`02Ool002=oo`03001o
ogoo07Uoo`03001oogoo07Uoo`03001oogoo009oo`008goo00<007ooOol0NGoo00<007ooOol0NGoo
00<007ooOol00Woo000SOol00`00Oomoo`1iOol00`00Oomoo`1iOol00`00Oomoo`02Ool001]oo`<0
00Eoo`03001oogoo0?Eoo`03001oogoo009oo`0077oo00<007ooOol017oo00<007ooOol00Woo0`00
97oo0`00bGoo00<007ooOol00Woo000LOol00`00Oomoo`04Ool30002Ool3000TOol30037Ool30004
Ool001aoo`03001oogoo00Aoo`03001oogoo009oo`<002Aoo`<00<Uoo`03001oogoo009oo`0077oo
00<007ooOol017oo00<007ooOol0mGoo00<007ooOol00Woo000KOol20006Ool00`00Oomoo`3eOol0
0`00Oomoo`02Ool002=oo`03001oogoo0?Eoo`03001oogoo009oo`008goo00<007ooOol0mGoo00<0
07ooOol00Woo000SOol00`00Oomoo`3eOol00`00Oomoo`02Ool002=oo`800?Eoo`8000Aoo`008goo
00<007ooOol0mGoo00<007ooOol00Woo000SOol00`00Oomoo`3eOol00`00Oomoo`02Ool002=oo`03
001oogoo0?Eoo`03001oogoo009oo`008goo00<007ooOol0mGoo00<007ooOol00Woo000SOol00`00
Oomoo`03Ool00`00Oomoo`0NOol00`00Oomoo`0NOol00`00Oomoo`0MOol00`00Oomoo`0NOol00`00
Oomoo`0NOol00`00Oomoo`0NOol00`00Oomoo`0MOol00`00Oomoo`0:Ool00`00Oomoo`02Ool002=o
ooT000Aoo`00\
\>"],
  ImageRangeCache->{{{0, 287}, {176.938, 0}} -> {-0.0319194, -0.173387, 
  0.000764225, 0.00719681}}]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{

Cell["\<\
Code for PlotFractionCorrectAtTimeTVersusStdDevError in here \
...\
\>", "Subsubsection"],

Cell["\<\
PlotFractionCorrectAtTimeTVersusStdDevError[bugStats_, t_]:=
\tModule[{data},
\t\tdata = Map[{ExtractStdDevError[#],
\t\t            FractionCorrectAtTimeT[#,t],
\t\t            StdErrorOfMeanFractionCorrectAtTimeT[#,t]}&, bugStats];
\t\tErrorListPlot[data,
\t\t              PlotRange->{0,1.1},
\t\t              Frame->True,
\t\t              FrameLabel->{\"stdDevError\", \"fraction \
correct\",\"\",\"\"}]
\t]

ExtractStdDevError[{stdDevError_,_,_,_,_,_,_,_,_,_}]:= stdDevError

(* Fraction correct at time t *)
FractionCorrectAtTimeT[{_, r_, _, _, w_, _, _, u_, _, _}, t_]:=
\tModule[{numerator},
\t\tnumerator = Length[Select[r, (#==t)&]];
\t\tIf[numerator>0, 
\t\t   N[numerator/Length[Select[Join[r,w,u], (#==t)&]]],
\t\t   0
\t\t  ]
\t]

StdErrorOfMeanFractionCorrectAtTimeT[{sDE_, 
                                      r_, mr_, sr_, 
                                      w_, mw_, sw_, 
                                      u_, mu_, su_}, t_]:=
\tModule[{f},
\t   f = FractionCorrectAtTimeT[{sDE, r, mr, sr, w, mw, sw, u, mu, su},t];
\t   If[f>0,
\t\t  Sqrt[(1-f) f/Length[Select[Join[r,w,u], (#==t)&]]],
\t\t  0
\t\t ]
\t]
\t\t
\
\>", "Input",
  InitializationCell->True]
}, Closed]]
}, Open  ]]
}, Open  ]]
},
FrontEndVersion->"Macintosh 3.0",
ScreenRectangle->{{0, 640}, {0, 460}},
AutoGeneratedPackage->None,
WindowSize->{636, 434},
WindowMargins->{{0, Automatic}, {Automatic, 1}},
PrintingCopies->1,
PrintingPageRange->{1, Automatic},
StyleDefinitions -> Notebook[{
  
  Cell[CellGroupData[{
  Cell["Style Definitions", "Subtitle"],
  
  Cell["\<\
Modify the definitions below to change the default appearance of \
all cells in a given style.  Make modifications to any definition using \
commands in the Format menu.\
\>", "Text"],
  
  Cell[CellGroupData[{
  
  Cell["Style Environment Names", "Section"],
  
  Cell[StyleData[All, "Working"],
    PageWidth->WindowWidth,
    ScriptMinSize->9],
  
  Cell[StyleData[All, "Presentation"],
    PageWidth->WindowWidth,
    ScriptMinSize->12,
    FontSize->16],
  
  Cell[StyleData[All, "Condensed"],
    PageWidth->WindowWidth,
    CellBracketOptions->{"Margins"->{1, 1},
    "Widths"->{0, 5}},
    ScriptMinSize->8,
    FontSize->11],
  
  Cell[StyleData[All, "Printout"],
    PageWidth->PaperWidth,
    ScriptMinSize->5,
    FontSize->10,
    PrivateFontOptions->{"FontType"->"Outline"}]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Notebook Options", "Section"],
  
  Cell["\<\
The options defined for the style below will be used at the \
Notebook level.\
\>", "Text"],
  
  Cell[StyleData["Notebook"],
    PageHeaders->{{Cell[ 
            TextData[ {
              CounterBox[ "Page"]}], "PageNumber"], None, Cell[ 
            TextData[ {
              ValueBox[ "FileName"]}], "Header"]}, {Cell[ 
            TextData[ {
              ValueBox[ "FileName"]}], "Header"], None, Cell[ 
            TextData[ {
              CounterBox[ "Page"]}], "PageNumber"]}},
    CellFrameLabelMargins->6,
    StyleMenuListing->None]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Styles for Headings", "Section"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Title"],
    CellMargins->{{12, Inherited}, {20, 40}},
    CellGroupingRules->{"TitleGrouping", 0},
    PageBreakBelow->False,
    CounterIncrements->"Title",
    CounterAssignments->{{"Section", 0}, {"Equation", 0}, {"Figure", 0}, {
        "Subtitle", 0}, {"Subsubtitle", 0}},
    FontFamily->"Helvetica",
    FontSize->36,
    FontWeight->"Bold"],
  
  Cell[StyleData["Title", "Presentation"],
    CellMargins->{{24, 10}, {20, 40}},
    LineSpacing->{1, 0},
    FontSize->44],
  
  Cell[StyleData["Title", "Condensed"],
    CellMargins->{{8, 10}, {4, 8}},
    FontSize->20],
  
  Cell[StyleData["Title", "Printout"],
    CellMargins->{{2, 10}, {12, 30}},
    FontSize->24]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Subtitle"],
    CellMargins->{{12, Inherited}, {20, 15}},
    CellGroupingRules->{"TitleGrouping", 10},
    PageBreakBelow->False,
    CounterIncrements->"Subtitle",
    CounterAssignments->{{"Section", 0}, {"Equation", 0}, {"Figure", 0}, {
        "Subsubtitle", 0}},
    FontFamily->"Helvetica",
    FontSize->24],
  
  Cell[StyleData["Subtitle", "Presentation"],
    CellMargins->{{24, 10}, {20, 20}},
    LineSpacing->{1, 0},
    FontSize->36],
  
  Cell[StyleData["Subtitle", "Condensed"],
    CellMargins->{{8, 10}, {4, 4}},
    FontSize->14],
  
  Cell[StyleData["Subtitle", "Printout"],
    CellMargins->{{2, 10}, {12, 8}},
    FontSize->18]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Subsubtitle"],
    CellMargins->{{12, Inherited}, {20, 15}},
    CellGroupingRules->{"TitleGrouping", 20},
    PageBreakBelow->False,
    CounterIncrements->"Subsubtitle",
    CounterAssignments->{{"Section", 0}, {"Equation", 0}, {"Figure", 0}},
    FontFamily->"Helvetica",
    FontSize->14,
    FontSlant->"Italic"],
  
  Cell[StyleData["Subsubtitle", "Presentation"],
    CellMargins->{{24, 10}, {20, 20}},
    LineSpacing->{1, 0},
    FontSize->24],
  
  Cell[StyleData["Subsubtitle", "Condensed"],
    CellMargins->{{8, 10}, {8, 8}},
    FontSize->12],
  
  Cell[StyleData["Subsubtitle", "Printout"],
    CellMargins->{{2, 10}, {12, 8}},
    FontSize->14]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Section"],
    CellDingbat->"\[FilledSquare]",
    CellMargins->{{25, Inherited}, {8, 24}},
    CellGroupingRules->{"SectionGrouping", 30},
    PageBreakBelow->False,
    CounterIncrements->"Section",
    CounterAssignments->{{"Subsection", 0}, {"Subsubsection", 0}},
    FontFamily->"Helvetica",
    FontSize->16,
    FontWeight->"Bold"],
  
  Cell[StyleData["Section", "Presentation"],
    CellMargins->{{40, 10}, {11, 32}},
    LineSpacing->{1, 0},
    FontSize->24],
  
  Cell[StyleData["Section", "Condensed"],
    CellMargins->{{18, Inherited}, {6, 12}},
    FontSize->12],
  
  Cell[StyleData["Section", "Printout"],
    CellMargins->{{13, 0}, {7, 22}},
    FontSize->14]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Subsection"],
    CellDingbat->"\[FilledSmallSquare]",
    CellMargins->{{22, Inherited}, {8, 20}},
    CellGroupingRules->{"SectionGrouping", 40},
    PageBreakBelow->False,
    CounterIncrements->"Subsection",
    CounterAssignments->{{"Subsubsection", 0}},
    FontSize->14,
    FontWeight->"Bold"],
  
  Cell[StyleData["Subsection", "Presentation"],
    CellMargins->{{36, 10}, {11, 32}},
    LineSpacing->{1, 0},
    FontSize->22],
  
  Cell[StyleData["Subsection", "Condensed"],
    CellMargins->{{16, Inherited}, {6, 12}},
    FontSize->12],
  
  Cell[StyleData["Subsection", "Printout"],
    CellMargins->{{9, 0}, {7, 22}},
    FontSize->12]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Subsubsection"],
    CellDingbat->"\[FilledSmallSquare]",
    CellMargins->{{22, Inherited}, {8, 18}},
    CellGroupingRules->{"SectionGrouping", 50},
    PageBreakBelow->False,
    CounterIncrements->"Subsubsection",
    FontWeight->"Bold",
    FontColor->RGBColor[1, 0, 0]],
  
  Cell[StyleData["Subsubsection", "Presentation"],
    CellMargins->{{34, 10}, {11, 26}},
    LineSpacing->{1, 0},
    FontSize->18],
  
  Cell[StyleData["Subsubsection", "Condensed"],
    CellMargins->{{17, Inherited}, {6, 12}},
    FontSize->10],
  
  Cell[StyleData["Subsubsection", "Printout"],
    CellMargins->{{9, 0}, {7, 14}},
    FontSize->11]
  }, Open  ]]
  }, Open  ]],
  
  Cell[CellGroupData[{
  
  Cell["Styles for Body Text", "Section"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Text"],
    CellMargins->{{12, 10}, {7, 7}},
    LineSpacing->{1, 3},
    CounterIncrements->"Text"],
  
  Cell[StyleData["Text", "Presentation"],
    CellMargins->{{24, 10}, {10, 10}},
    LineSpacing->{1, 5}],
  
  Cell[StyleData["Text", "Condensed"],
    CellMargins->{{8, 10}, {6, 6}},
    LineSpacing->{1, 1}],
  
  Cell[StyleData["Text", "Printout"],
    CellMargins->{{2, 2}, {6, 6}}]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["SmallText"],
    CellMargins->{{12, 10}, {6, 6}},
    LineSpacing->{1, 3},
    CounterIncrements->"SmallText",
    FontFamily->"Helvetica",
    FontSize->9],
  
  Cell[StyleData["SmallText", "Presentation"],
    CellMargins->{{24, 10}, {8, 8}},
    LineSpacing->{1, 5},
    FontSize->12],
  
  Cell[StyleData["SmallText", "Condensed"],
    CellMargins->{{8, 10}, {5, 5}},
    LineSpacing->{1, 2},
    FontSize->9],
  
  Cell[StyleData["SmallText", "Printout"],
    CellMargins->{{2, 2}, {5, 5}},
    FontSize->7]
  }, Closed]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Styles for Input/Output", "Section"],
  
  Cell["\<\
The cells in this section define styles used for input and output \
to the kernel.  Be careful when modifying, renaming, or removing these \
styles, because the front end associates special meanings with these style \
names. Some attributes for these styles are actually set in FormatType Styles \
(in the last section of this stylesheet). \
\>", "Text"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Input"],
    CellMargins->{{45, 10}, {5, 7}},
    Evaluatable->True,
    CellGroupingRules->"InputGrouping",
    PageBreakWithin->False,
    GroupPageBreakWithin->False,
    CellLabelMargins->{{11, Inherited}, {Inherited, Inherited}},
    DefaultFormatType->DefaultInputFormatType,
    AutoItalicWords->{},
    FormatType->InputForm,
    ShowStringCharacters->True,
    NumberMarks->True,
    CounterIncrements->"Input",
    FontSize->10,
    FontWeight->"Bold"],
  
  Cell[StyleData["Input", "Presentation"],
    CellMargins->{{72, Inherited}, {8, 10}},
    LineSpacing->{1, 0}],
  
  Cell[StyleData["Input", "Condensed"],
    CellMargins->{{40, 10}, {2, 3}}],
  
  Cell[StyleData["Input", "Printout"],
    CellMargins->{{39, 0}, {4, 6}},
    FontSize->9]
  }, Closed]],
  
  Cell[StyleData["InputOnly"],
    Evaluatable->True,
    CellGroupingRules->"InputGrouping",
    DefaultFormatType->DefaultInputFormatType,
    AutoItalicWords->{},
    FormatType->InputForm,
    ShowStringCharacters->True,
    NumberMarks->True,
    CounterIncrements->"Input",
    StyleMenuListing->None,
    FontWeight->"Bold"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Output"],
    CellMargins->{{47, 10}, {7, 5}},
    CellEditDuplicate->True,
    CellGroupingRules->"OutputGrouping",
    CellHorizontalScrolling->True,
    PageBreakWithin->False,
    GroupPageBreakWithin->False,
    GeneratedCell->True,
    CellAutoOverwrite->True,
    CellLabelMargins->{{11, Inherited}, {Inherited, Inherited}},
    DefaultFormatType->DefaultOutputFormatType,
    AutoItalicWords->{},
    FormatType->InputForm,
    CounterIncrements->"Output",
    FontSize->10],
  
  Cell[StyleData["Output", "Presentation"],
    CellMargins->{{72, Inherited}, {10, 8}},
    LineSpacing->{1, 0}],
  
  Cell[StyleData["Output", "Condensed"],
    CellMargins->{{41, Inherited}, {3, 2}}],
  
  Cell[StyleData["Output", "Printout"],
    CellMargins->{{39, 0}, {6, 4}},
    FontSize->9]
  }, Open  ]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Message"],
    CellMargins->{{45, Inherited}, {Inherited, Inherited}},
    CellGroupingRules->"OutputGrouping",
    PageBreakWithin->False,
    GroupPageBreakWithin->False,
    GeneratedCell->True,
    CellAutoOverwrite->True,
    ShowCellLabel->False,
    CellLabelMargins->{{11, Inherited}, {Inherited, Inherited}},
    DefaultFormatType->DefaultOutputFormatType,
    AutoItalicWords->{},
    FormatType->InputForm,
    CounterIncrements->"Message",
    StyleMenuListing->None,
    FontColor->RGBColor[0, 0, 1]],
  
  Cell[StyleData["Message", "Presentation"],
    CellMargins->{{72, Inherited}, {Inherited, Inherited}},
    LineSpacing->{1, 0}],
  
  Cell[StyleData["Message", "Condensed"],
    CellMargins->{{41, Inherited}, {Inherited, Inherited}}],
  
  Cell[StyleData["Message", "Printout"],
    CellMargins->{{39, Inherited}, {Inherited, Inherited}},
    FontSize->8,
    FontColor->GrayLevel[0]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Print"],
    CellMargins->{{45, Inherited}, {Inherited, Inherited}},
    CellGroupingRules->"OutputGrouping",
    PageBreakWithin->False,
    GroupPageBreakWithin->False,
    GeneratedCell->True,
    CellAutoOverwrite->True,
    ShowCellLabel->False,
    CellLabelMargins->{{11, Inherited}, {Inherited, Inherited}},
    DefaultFormatType->DefaultOutputFormatType,
    AutoItalicWords->{},
    FormatType->InputForm,
    CounterIncrements->"Print",
    StyleMenuListing->None,
    FontSize->10],
  
  Cell[StyleData["Print", "Presentation"],
    CellMargins->{{72, Inherited}, {Inherited, Inherited}},
    LineSpacing->{1, 0}],
  
  Cell[StyleData["Print", "Condensed"],
    CellMargins->{{41, Inherited}, {Inherited, Inherited}}],
  
  Cell[StyleData["Print", "Printout"],
    CellMargins->{{39, Inherited}, {Inherited, Inherited}},
    FontSize->8]
  }, Open  ]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Graphics"],
    CellMargins->{{4, Inherited}, {Inherited, Inherited}},
    CellGroupingRules->"GraphicsGrouping",
    CellHorizontalScrolling->True,
    PageBreakWithin->False,
    GeneratedCell->True,
    CellAutoOverwrite->True,
    ShowCellLabel->False,
    DefaultFormatType->DefaultOutputFormatType,
    FormatType->InputForm,
    CounterIncrements->"Graphics",
    ImageMargins->{{43, Inherited}, {Inherited, 0}},
    StyleMenuListing->None],
  
  Cell[StyleData["Graphics", "Presentation"],
    ImageMargins->{{62, Inherited}, {Inherited, 0}}],
  
  Cell[StyleData["Graphics", "Condensed"],
    ImageMargins->{{38, Inherited}, {Inherited, 0}},
    Magnification->0.6],
  
  Cell[StyleData["Graphics", "Printout"],
    ImageMargins->{{30, Inherited}, {Inherited, 0}},
    FontSize->9,
    Magnification->0.8]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["CellLabel"],
    StyleMenuListing->None,
    FontFamily->"Helvetica",
    FontSize->9,
    FontColor->RGBColor[0, 0, 1]],
  
  Cell[StyleData["CellLabel", "Presentation"],
    FontSize->12],
  
  Cell[StyleData["CellLabel", "Condensed"],
    FontSize->9],
  
  Cell[StyleData["CellLabel", "Printout"],
    FontFamily->"Courier",
    FontSize->8,
    FontSlant->"Italic",
    FontColor->GrayLevel[0]]
  }, Closed]]
  }, Open  ]],
  
  Cell[CellGroupData[{
  
  Cell["Formulas and Programming", "Section"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["InlineFormula"],
    CellMargins->{{10, 4}, {0, 8}},
    CellHorizontalScrolling->True,
    ScriptLevel->1,
    SingleLetterItalics->True],
  
  Cell[StyleData["InlineFormula", "Presentation"],
    CellMargins->{{24, 10}, {10, 10}},
    LineSpacing->{1, 5}],
  
  Cell[StyleData["InlineFormula", "Condensed"],
    CellMargins->{{8, 10}, {6, 6}},
    LineSpacing->{1, 1}],
  
  Cell[StyleData["InlineFormula", "Printout"],
    CellMargins->{{2, 0}, {6, 6}}]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["DisplayFormula"],
    CellMargins->{{42, Inherited}, {Inherited, Inherited}},
    CellHorizontalScrolling->True,
    DefaultFormatType->DefaultInputFormatType,
    ScriptLevel->0,
    SingleLetterItalics->True,
    UnderoverscriptBoxOptions->{LimitsPositioning->True}],
  
  Cell[StyleData["DisplayFormula", "Presentation"],
    LineSpacing->{1, 5}],
  
  Cell[StyleData["DisplayFormula", "Condensed"],
    LineSpacing->{1, 1}],
  
  Cell[StyleData["DisplayFormula", "Printout"]]
  }, Closed]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Styles for Headers and Footers", "Section"],
  
  Cell[StyleData["Header"],
    CellMargins->{{0, 0}, {4, 1}},
    StyleMenuListing->None,
    FontSize->10,
    FontSlant->"Italic"],
  
  Cell[StyleData["Footer"],
    CellMargins->{{0, 0}, {0, 4}},
    StyleMenuListing->None,
    FontSize->9,
    FontSlant->"Italic"],
  
  Cell[StyleData["PageNumber"],
    CellMargins->{{0, 0}, {4, 1}},
    StyleMenuListing->None,
    FontFamily->"Times",
    FontSize->10]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Palette Styles", "Section"],
  
  Cell["\<\
The cells below define styles that define standard \
ButtonFunctions, for use in palette buttons.\
\>", "Text"],
  
  Cell[StyleData["Paste"],
    StyleMenuListing->None,
    ButtonStyleMenuListing->Automatic,
    ButtonBoxOptions->{ButtonFunction:>(FrontEndExecute[ {
        FrontEnd`NotebookApply[ 
          FrontEnd`InputNotebook[ ], #, After]}]&)}],
  
  Cell[StyleData["Evaluate"],
    StyleMenuListing->None,
    ButtonStyleMenuListing->Automatic,
    ButtonBoxOptions->{ButtonFunction:>(FrontEndExecute[ {
        FrontEnd`NotebookApply[ 
          FrontEnd`InputNotebook[ ], #, All], 
        SelectionEvaluate[ 
          FrontEnd`InputNotebook[ ], All]}]&)}],
  
  Cell[StyleData["EvaluateCell"],
    StyleMenuListing->None,
    ButtonStyleMenuListing->Automatic,
    ButtonBoxOptions->{ButtonFunction:>(FrontEndExecute[ {
        FrontEnd`NotebookApply[ 
          FrontEnd`InputNotebook[ ], #, All], 
        FrontEnd`SelectionMove[ 
          FrontEnd`InputNotebook[ ], All, Cell, 1], 
        FrontEnd`SelectionEvaluateCreateCell[ 
          FrontEnd`InputNotebook[ ], All]}]&)}],
  
  Cell[StyleData["CopyEvaluate"],
    StyleMenuListing->None,
    ButtonStyleMenuListing->Automatic,
    ButtonBoxOptions->{ButtonFunction:>(FrontEndExecute[ {
        FrontEnd`SelectionCreateCell[ 
          FrontEnd`InputNotebook[ ], All], 
        FrontEnd`NotebookApply[ 
          FrontEnd`InputNotebook[ ], #, All], 
        FrontEnd`SelectionEvaluate[ 
          FrontEnd`InputNotebook[ ], All]}]&)}],
  
  Cell[StyleData["CopyEvaluateCell"],
    StyleMenuListing->None,
    ButtonStyleMenuListing->Automatic,
    ButtonBoxOptions->{ButtonFunction:>(FrontEndExecute[ {
        FrontEnd`SelectionCreateCell[ 
          FrontEnd`InputNotebook[ ], All], 
        FrontEnd`NotebookApply[ 
          FrontEnd`InputNotebook[ ], #, All], 
        FrontEnd`SelectionEvaluateCreateCell[ 
          FrontEnd`InputNotebook[ ], All]}]&)}]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Hyperlink Styles", "Section"],
  
  Cell["\<\
The cells below define styles useful for making hypertext \
ButtonBoxes.  The \"Hyperlink\" style is for links within the same Notebook, \
or between Notebooks.\
\>", "Text"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Hyperlink"],
    StyleMenuListing->None,
    ButtonStyleMenuListing->Automatic,
    FontColor->RGBColor[0, 0, 1],
    FontVariations->{"Underline"->True},
    ButtonBoxOptions->{ButtonFunction:>(FrontEndExecute[ {
        FrontEnd`NotebookLocate[ #2]}]&),
    Active->True,
    ButtonNote->ButtonData}],
  
  Cell[StyleData["Hyperlink", "Presentation"]],
  
  Cell[StyleData["Hyperlink", "Condensed"]],
  
  Cell[StyleData["Hyperlink", "Printout"],
    FontColor->GrayLevel[0],
    FontVariations->{"Underline"->False}]
  }, Closed]],
  
  Cell["\<\
The following styles are for linking automatically to the on-line \
help system.\
\>", "Text"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["MainBookLink"],
    StyleMenuListing->None,
    ButtonStyleMenuListing->Automatic,
    FontColor->RGBColor[0, 0, 1],
    FontVariations->{"Underline"->True},
    ButtonBoxOptions->{ButtonFunction:>(FrontEndExecute[ {
        FrontEnd`HelpBrowserLookup[ "MainBook", #]}]&),
    Active->True,
    ButtonFrame->"None"}],
  
  Cell[StyleData["MainBookLink", "Presentation"]],
  
  Cell[StyleData["MainBookLink", "Condensed"]],
  
  Cell[StyleData["MainBookLink", "Printout"],
    FontColor->GrayLevel[0],
    FontVariations->{"Underline"->False}]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["AddOnsLink"],
    StyleMenuListing->None,
    ButtonStyleMenuListing->Automatic,
    FontFamily->"Courier",
    FontColor->RGBColor[0, 0, 1],
    FontVariations->{"Underline"->True},
    ButtonBoxOptions->{ButtonFunction:>(FrontEndExecute[ {
        FrontEnd`HelpBrowserLookup[ "AddOns", #]}]&),
    Active->True,
    ButtonFrame->"None"}],
  
  Cell[StyleData["AddOnsLink", "Presentation"]],
  
  Cell[StyleData["AddOnsLink", "Condensed"]],
  
  Cell[StyleData["AddOnLink", "Printout"],
    FontColor->GrayLevel[0],
    FontVariations->{"Underline"->False}]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["RefGuideLink"],
    StyleMenuListing->None,
    ButtonStyleMenuListing->Automatic,
    FontFamily->"Courier",
    FontColor->RGBColor[0, 0, 1],
    FontVariations->{"Underline"->True},
    ButtonBoxOptions->{ButtonFunction:>(FrontEndExecute[ {
        FrontEnd`HelpBrowserLookup[ "RefGuideLink", #]}]&),
    Active->True,
    ButtonFrame->"None"}],
  
  Cell[StyleData["RefGuideLink", "Presentation"]],
  
  Cell[StyleData["RefGuideLink", "Condensed"]],
  
  Cell[StyleData["RefGuideLink", "Printout"],
    FontColor->GrayLevel[0],
    FontVariations->{"Underline"->False}]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["GettingStartedLink"],
    StyleMenuListing->None,
    ButtonStyleMenuListing->Automatic,
    FontColor->RGBColor[0, 0, 1],
    FontVariations->{"Underline"->True},
    ButtonBoxOptions->{ButtonFunction:>(FrontEndExecute[ {
        FrontEnd`HelpBrowserLookup[ "GettingStarted", #]}]&),
    Active->True,
    ButtonFrame->"None"}],
  
  Cell[StyleData["GettingStartedLink", "Presentation"]],
  
  Cell[StyleData["GettingStartedLink", "Condensed"]],
  
  Cell[StyleData["GettingStartedLink", "Printout"],
    FontColor->GrayLevel[0],
    FontVariations->{"Underline"->False}]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["OtherInformationLink"],
    StyleMenuListing->None,
    ButtonStyleMenuListing->Automatic,
    FontColor->RGBColor[0, 0, 1],
    FontVariations->{"Underline"->True},
    ButtonBoxOptions->{ButtonFunction:>(FrontEndExecute[ {
        FrontEnd`HelpBrowserLookup[ "OtherInformation", #]}]&),
    Active->True,
    ButtonFrame->"None"}],
  
  Cell[StyleData["OtherInformationLink", "Presentation"]],
  
  Cell[StyleData["OtherInformationLink", "Condensed"]],
  
  Cell[StyleData["OtherInformationLink", "Printout"],
    FontColor->GrayLevel[0],
    FontVariations->{"Underline"->False}]
  }, Closed]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["Placeholder Styles", "Section"],
  
  Cell["\<\
The cells below define styles useful for making placeholder \
objects in palette templates.\
\>", "Text"],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["Placeholder"],
    Editable->False,
    Selectable->False,
    StyleBoxAutoDelete->True,
    Placeholder->True,
    StyleMenuListing->None],
  
  Cell[StyleData["Placeholder", "Presentation"]],
  
  Cell[StyleData["Placeholder", "Condensed"]],
  
  Cell[StyleData["Placeholder", "Printout"]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell[StyleData["SelectionPlaceholder"],
    Editable->False,
    Selectable->False,
    StyleBoxAutoDelete->True,
    Placeholder->Primary,
    StyleMenuListing->None,
    DrawHighlighted->True],
  
  Cell[StyleData["SelectionPlaceholder", "Presentation"]],
  
  Cell[StyleData["SelectionPlaceholder", "Condensed"]],
  
  Cell[StyleData["SelectionPlaceholder", "Printout"]]
  }, Closed]]
  }, Closed]],
  
  Cell[CellGroupData[{
  
  Cell["FormatType Styles", "Section"],
  
  Cell["\<\
The cells below define styles that are mixed in with the styles \
of most cells.  If a cell's FormatType matches the name of one of the styles \
defined below, then that style is applied between the cell's style and its \
own options. This is particularly true of Input and Output.\
\>", "Text"],
  
  Cell[StyleData["CellExpression"],
    PageWidth->Infinity,
    CellMargins->{{6, Inherited}, {Inherited, Inherited}},
    ShowCellLabel->False,
    ShowSpecialCharacters->False,
    AllowInlineCells->False,
    AutoItalicWords->{},
    StyleMenuListing->None,
    FontFamily->"Courier",
    FontSize->12,
    Background->GrayLevel[1]],
  
  Cell[StyleData["InputForm"],
    AllowInlineCells->False,
    StyleMenuListing->None,
    FontFamily->"Courier"],
  
  Cell[StyleData["OutputForm"],
    PageWidth->Infinity,
    TextAlignment->Left,
    LineSpacing->{0.6, 1},
    StyleMenuListing->None,
    FontFamily->"Courier"],
  
  Cell[StyleData["StandardForm"],
    LineSpacing->{1.25, 0},
    StyleMenuListing->None,
    FontFamily->"Courier"],
  
  Cell[StyleData["TraditionalForm"],
    LineSpacing->{1.25, 0},
    SingleLetterItalics->True,
    TraditionalFunctionNotation->True,
    DelimiterMatching->None,
    StyleMenuListing->None],
  
  Cell["\<\
The style defined below is mixed in to any cell that is in an \
inline cell within another.\
\>", "Text"],
  
  Cell[StyleData["InlineCell"],
    TextAlignment->Left,
    ScriptLevel->1,
    StyleMenuListing->None],
  
  Cell[StyleData["InlineCellEditing"],
    StyleMenuListing->None,
    Background->RGBColor[1, 0.749996, 0.8]]
  }, Closed]]
  }, Open  ]]
  }],
MacintoshSystemPageSetup->"\<\
00<0004/0B`000002n88o?mooh<<H`Ui0fL5:0?l0080004/0B`000000]P2:001
0000I00000400`<300000BL?00400@00000000000000060801T1T000000@0000
00000000004000000000000000000000\>"
]


(***********************************************************************
Cached data follows.  If you edit this Notebook file directly, not using
Mathematica, you must remove the line containing CacheID at the top of 
the file.  The cache data will then be recreated when you save this file 
from within Mathematica.
***********************************************************************)

(*CellTagsOutline
CellTagsIndex->{}
*)

(*CellTagsIndex
CellTagsIndex->{}
*)

(*NotebookFileOutline
Notebook[{

Cell[CellGroupData[{
Cell[1731, 51, 110, 2, 136, "Title",
  Evaluatable->False],
Cell[1844, 55, 62, 0, 53, "Subsubtitle"],
Cell[1909, 57, 263, 8, 90, "Input",
  InitializationCell->True],

Cell[CellGroupData[{
Cell[2197, 69, 102, 3, 50, "Section",
  Evaluatable->False],
Cell[2302, 74, 927, 19, 206, "Text",
  Evaluatable->False]
}, Open  ]],

Cell[CellGroupData[{
Cell[3266, 98, 31, 0, 50, "Section"],
Cell[3300, 100, 1222, 17, 174, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[4559, 122, 33, 0, 50, "Section"],
Cell[4595, 124, 526, 12, 82, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[5158, 141, 63, 0, 50, "Section"],
Cell[5224, 143, 343, 6, 62, "Text"],

Cell[CellGroupData[{
Cell[5592, 153, 32, 0, 42, "Subsubsection"],
Cell[5627, 155, 226, 4, 46, "Text"],
Cell[5856, 161, 47, 1, 25, "Input"],
Cell[5906, 164, 434, 7, 78, "Text"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[6389, 177, 59, 0, 50, "Section"],
Cell[6451, 179, 2134, 31, 318, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[8622, 215, 53, 0, 50, "Section"],
Cell[8678, 217, 1660, 26, 286, "Text"],
Cell[10341, 245, 459, 13, 155, "Input",
  InitializationCell->True],
Cell[10803, 260, 253, 5, 46, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[11093, 270, 63, 0, 50, "Section"],
Cell[11159, 272, 596, 13, 174, "Text"],
Cell[11758, 287, 632, 16, 190, "Text"],
Cell[12393, 305, 105, 3, 30, "Text"],

Cell[CellGroupData[{
Cell[12523, 312, 41, 0, 25, "Input"],
Cell[12567, 314, 1156, 27, 177, "Print"]
}, Open  ]],

Cell[CellGroupData[{
Cell[13760, 346, 32, 0, 42, "Subsubsection"],
Cell[13795, 348, 1274, 20, 206, "Text"],

Cell[CellGroupData[{
Cell[15094, 372, 236, 7, 90, "Input"],
Cell[15333, 381, 1011, 27, 346, "Print"],
Cell[16347, 410, 379, 13, 39, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[16775, 429, 32, 0, 42, "Subsubsection"],
Cell[16810, 431, 394, 6, 78, "Text"],

Cell[CellGroupData[{
Cell[17229, 441, 235, 7, 90, "Input"],
Cell[17467, 450, 1008, 27, 346, "Print"],
Cell[18478, 479, 372, 12, 39, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[18899, 497, 69, 0, 42, "Subsubsection"],
Cell[18971, 499, 3745, 81, 1026, "Input",
  InitializationCell->True],
Cell[22719, 582, 988, 14, 142, "Text"],

Cell[CellGroupData[{
Cell[23732, 600, 67, 0, 25, "Input"],
Cell[23802, 602, 62, 4, 24, "Output"]
}, Open  ]],
Cell[23879, 609, 690, 11, 110, "Text"]
}, Open  ]],

Cell[CellGroupData[{
Cell[24606, 625, 72, 0, 42, "Subsubsection"],
Cell[24681, 627, 1951, 49, 623, "Input",
  InitializationCell->True],
Cell[26635, 678, 1062, 26, 324, "Input",
  InitializationCell->True],
Cell[27700, 706, 205, 8, 90, "Input",
  InitializationCell->True]
}, Closed]]
}, Open  ]],

Cell[CellGroupData[{
Cell[27954, 720, 70, 0, 30, "Section"],
Cell[28027, 722, 578, 9, 94, "Text"],

Cell[CellGroupData[{
Cell[28630, 735, 38, 0, 25, "Input"],
Cell[28671, 737, 492, 12, 86, "Print"]
}, Open  ]],

Cell[CellGroupData[{
Cell[29200, 754, 32, 0, 42, "Subsubsection"],
Cell[29235, 756, 598, 9, 94, "Text"],

Cell[CellGroupData[{
Cell[29858, 769, 163, 4, 51, "Input"],
Cell[30024, 775, 1333, 36, 463, "Print"],
Cell[31360, 813, 1750, 50, 160, "Output"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[33159, 869, 66, 0, 42, "Subsubsection"],
Cell[33228, 871, 978, 18, 220, "Input",
  InitializationCell->True]
}, Closed]],

Cell[CellGroupData[{
Cell[34243, 894, 32, 0, 28, "Subsubsection"],
Cell[34278, 896, 217, 4, 46, "Text"],

Cell[CellGroupData[{
Cell[34520, 904, 89, 3, 25, "Input"],
Cell[34612, 909, 26520, 825, 186, 8520, 598, "GraphicsData", 
"PostScript", "Graphics"]
}, Open  ]]
}, Open  ]],

Cell[CellGroupData[{
Cell[61181, 1740, 99, 3, 42, "Subsubsection"],
Cell[61283, 1745, 1194, 39, 480, "Input",
  InitializationCell->True]
}, Closed]]
}, Open  ]]
}, Open  ]]
}
]
*)




(***********************************************************************
End of Mathematica Notebook file.
***********************************************************************)

